<?php
// Ensure PHP 8.2 compatibility
declare(strict_types=1);

// Start session immediately for admin and user tracking
session_start();

// --- CONFIGURATION & SETUP CHECK ---
const DATA_DIR = 'data/';
const CONFIG_FILE = DATA_DIR . 'config.json';

if (!file_exists(CONFIG_FILE)) {
    die("Site Configuration Missing. Please upload and run <a href='setup.php'>setup.php</a> first.");
}

// Load Configuration
$config = json_decode(file_get_contents(CONFIG_FILE), true, 512, JSON_THROW_ON_ERROR);

// --- CONSTANTS & SETTINGS ---
// Debug mode (controlled via $config if you wanted, but strictly false for prod recommended)
const DEBUG_MODE = false; 
if (DEBUG_MODE) {
    ini_set('display_errors', '1'); ini_set('display_startup_errors', '1'); error_reporting(E_ALL);
} else {
    ini_set('display_errors', '0'); error_reporting(E_ALL);
}

const POSTS_PER_MINUTE = 5;
const POSTS_PER_DAY = 50; // Increased slightly for active users
const MAX_CONTENT_LENGTH = 4096; // Increased to 4KB
const POSTS_DIR = 'posts/';
const IMAGES_DIR = 'images/';

// --- CORE FUNCTIONS ---

// Robust Error Logging
function log_error(string $message): void {
    $debug_file = DATA_DIR . 'debug.txt';
    // Simple log rotation if it gets too big (5MB)
    if (file_exists($debug_file) && filesize($debug_file) > 5 * 1024 * 1024) {
        rename($debug_file, DATA_DIR . 'debug_' . date('YmdHis') . '.bak');
    }
    error_log(date('Y-m-d H:i:s') . " - $message\n", 3, $debug_file);
}

// Fatal Error Handler
register_shutdown_function(function (): void {
    $error = error_get_last();
    if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR], true)) {
        log_error("Fatal Error: " . $error['message'] . " in " . $error['file'] . ":" . $error['line']);
        if (!headers_sent()) { echo "Server error. Checked logs."; }
    }
});

// Environment Checks
if (ini_get('memory_limit') < '128M') { @ini_set('memory_limit', '256M'); }
$required_extensions = ['gd', 'json', 'mbstring'];
foreach ($required_extensions as $ext) {
    if (!extension_loaded($ext)) { die("Server configuration error: Missing PHP extension '$ext'."); }
}

// Ensure Directories
foreach ([DATA_DIR, POSTS_DIR, IMAGES_DIR] as $dir) {
    if (!is_dir($dir) && !mkdir($dir, 0755, true)) { die("Fatal: Could not create directory $dir"); }
}

// --- PRIVACY & IDENTITY ---

// New Secure Identity Function replacing ip_to_alpha
function get_poster_identity(string $ip): string {
    global $config;
    if (($config['privacy_mode'] ?? 'hash') === 'anonymous') {
        return 'Anonymous';
    }
    // HMAC-SHA256 the IP with the site secret. This is irreversible without the secret.
    // We take the first 10 chars of the base64 output for a readable, unique "tripcode".
    $hash = hash_hmac('sha256', $ip, $config['hmac_secret']);
    return substr(str_replace(['+', '/', '='], '', base64_encode(pack('H*', $hash))), 0, 10);
}

// Standard Rate Limiting
function check_rate_limit(string $ip): bool {
    $rate_file = DATA_DIR . 'rate_' . md5($ip) . '.json';
    $now = time();
    $rates = file_exists($rate_file) ? (json_decode(file_get_contents($rate_file), true) ?: []) : [];
    // Filter old entries (older than 1 day)
    $rates = array_filter($rates, fn($t) => $t > ($now - 86400));
    
    $minute_count = count(array_filter($rates, fn($t) => $t > ($now - 60)));
    if ($minute_count >= POSTS_PER_MINUTE || count($rates) >= POSTS_PER_DAY) {
        return false;
    }
    $rates[] = $now;
    file_put_contents($rate_file, json_encode($rates));
    return true;
}

// ASCII Art Converter (Preserved Original Logic)
function image_to_ascii(string $image_path): string {
    try {
        $img = @imagecreatefromstring(file_get_contents($image_path));
        if (!$img) throw new Exception("Invalid image format");
        
        $w = imagesx($img); $h = imagesy($img);
        $new_w = 80; // Slightly higher resolution
        $new_h = (int)(($h / $w) * $new_w * 0.55); // Adjust aspect ratio for text char height

        $resized = imagecreatetruecolor($new_w, $new_h);
        imagecopyresampled($resized, $img, 0, 0, 0, 0, $new_w, $new_h, $w, $h);
        
        $chars = ['@', '%', '#', '*', '+', '=', '-', ':', '.', ' '];
        $c_count = count($chars);
        
        $ascii_img = imagecreatetruecolor($new_w * 8, $new_h * 14);
        $bg = imagecolorallocate($ascii_img, 0, 0, 0);         // Black background for better contrast
        $fg = imagecolorallocate($ascii_img, 255, 255, 255);   // White text
        imagefill($ascii_img, 0, 0, $bg);
        
        for ($y = 0; $y < $new_h; $y++) {
            for ($x = 0; $x < $new_w; $x++) {
                $rgb = imagecolorat($resized, $x, $y);
                $r = ($rgb >> 16) & 0xFF; $g = ($rgb >> 8) & 0xFF; $b = $rgb & 0xFF;
                $gray = ($r + $g + $b) / 3;
                $char_idx = (int)(($gray / 255) * ($c_count - 1));
                imagestring($ascii_img, 3, $x * 8, $y * 14, $chars[$c_count - 1 - $char_idx], $fg);
            }
        }
        
        $out_path = IMAGES_DIR . uniqid('ascii_', true) . '.png';
        imagepng($ascii_img, $out_path, 9);
        imagedestroy($img); imagedestroy($resized); imagedestroy($ascii_img);
        unlink($image_path);
        return $out_path;
    } catch (Exception $e) {
        log_error("ASCII Convert Fail: " . $e->getMessage());
        return '';
    }
}

// View Counting
function update_channel_views(string $channel_name, string $ip): array {
    $file = DATA_DIR . 'views_' . md5($channel_name) . '.json';
    $views = file_exists($file) ? json_decode(file_get_contents($file), true) : ['total' => 0, 'ips' => []];
    $views['total']++;
    // Hash IP before storing for unique views to respect privacy settings somewhat
    $ip_hash = md5($ip . ($config['hmac_secret'] ?? ''));
    if (!in_array($ip_hash, $views['ips'])) {
        $views['ips'][] = $ip_hash;
    }
    file_put_contents($file, json_encode($views));
    return ['total' => $views['total'], 'unique' => count($views['ips'])];
}

// --- DATA LOADING ---
$channels_file = DATA_DIR . 'channels.json';
$channels = file_exists($channels_file) ? json_decode(file_get_contents($channels_file), true) : [];

// --- AUTHENTICATION & ADMIN ACTIONS ---
$is_admin = isset($_SESSION['admin']) && $_SESSION['admin'] === true;
$user_id = $_SESSION['user_id'] ?? ($_SESSION['user_id'] = uniqid('usr_', true));

// Admin Login
if (isset($_POST['admin_login'], $_POST['password'])) {
    if (password_verify($_POST['password'], $config['admin_hash'])) {
        $_SESSION['admin'] = true;
        $is_admin = true;
        log_error("Admin logged in from " . $_SERVER['REMOTE_ADDR']);
        header("Location: " . $_SERVER['PHP_SELF']); exit;
    } else {
        $error = "Invalid Admin Password";
        log_error("Failed admin login from " . $_SERVER['REMOTE_ADDR']);
    }
}
// Admin Logout
if (isset($_GET['logout'])) {
    unset($_SESSION['admin']);
    session_destroy();
    header("Location: index.php"); exit;
}

// --- NEW ADMIN PANEL HANDLERS ---
if ($is_admin && $_SERVER['REQUEST_METHOD'] === 'POST') {
    // 1. Update General Settings
    if (isset($_POST['update_settings'])) {
        $config['site_name'] = htmlspecialchars($_POST['site_name'], ENT_QUOTES);
        $config['site_slogan'] = htmlspecialchars($_POST['site_slogan'], ENT_QUOTES);
        $config['site_description'] = htmlspecialchars($_POST['site_description'], ENT_QUOTES);
        $config['privacy_mode'] = $_POST['privacy_mode'];
        if (!empty($_POST['new_password'])) {
            $config['admin_hash'] = password_hash($_POST['new_password'], PASSWORD_DEFAULT);
        }
        file_put_contents(CONFIG_FILE, json_encode($config, JSON_PRETTY_PRINT));
        $success = "Settings Saved.";
    }
    // 2. Update Theme
    if (isset($_POST['update_theme'])) {
        foreach ($config['theme'] as $key => $val) {
            if (isset($_POST["theme_$key"])) $config['theme'][$key] = $_POST["theme_$key"];
        }
        file_put_contents(CONFIG_FILE, json_encode($config, JSON_PRETTY_PRINT));
        $success = "Theme Updated.";
    }
    // 3. Update About Page
    if (isset($_POST['update_about'])) {
        file_put_contents(DATA_DIR . 'about.json', json_encode([
            'title' => $_POST['about_title'],
            'content' => $_POST['about_content'] // HTML allowed here for rich text
        ], JSON_PRETTY_PRINT));
        $success = "About Page Updated.";
    }
    // 4. Manage Channels (Create/Delete)
    if (isset($_POST['create_channel'])) {
        $channels[] = [
            'name' => htmlspecialchars($_POST['channel_name']),
            'description' => htmlspecialchars($_POST['channel_desc']),
            'disclosure' => htmlspecialchars($_POST['channel_disc'])
        ];
        file_put_contents($channels_file, json_encode($channels, JSON_PRETTY_PRINT));
        $success = "Channel Created.";
    }
    if (isset($_POST['delete_channel'])) {
        $idx = (int)$_POST['channel_index'];
        if (isset($channels[$idx])) {
            unset($channels[$idx]);
            $channels = array_values($channels);
            file_put_contents($channels_file, json_encode($channels, JSON_PRETTY_PRINT));
            $success = "Channel Deleted.";
        }
    }
}

// --- POST/COMMENT HANDLERS (With Privacy & Security Updates) ---
function save_channel_posts(string $channel, array $posts): void {
    $file = POSTS_DIR . preg_replace('/[^a-z0-9_-]/i', '', $channel) . '.json';
    $fp = fopen($file, 'w');
    if (flock($fp, LOCK_EX)) {
        fwrite($fp, json_encode($posts, JSON_PRETTY_PRINT));
        flock($fp, LOCK_UN);
    }
    fclose($fp);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $ip = $_SERVER['REMOTE_ADDR'];

    // Handle Deletions (Admin or User's own session)
    if (isset($_POST['delete_post'], $_POST['channel'], $_POST['post_id'])) {
        $c_name = $_POST['channel'];
        $p_file = POSTS_DIR . preg_replace('/[^a-z0-9_-]/i', '', $c_name) . '.json';
        if (file_exists($p_file)) {
            $posts = json_decode(file_get_contents($p_file), true) ?: [];
            $initial_count = count($posts);
            $posts = array_filter($posts, function($p) use ($is_admin, $user_id) {
                return !($p['id'] === $_POST['post_id'] && ($is_admin || $p['user_session'] === $user_id));
            });
            if (count($posts) < $initial_count) {
                save_channel_posts($c_name, array_values($posts));
                header("Location: ?channel=" . urlencode($c_name)); exit;
            } else {
                $error = "Could not delete post (Unauthorized or not found).";
            }
        }
    }

    // NEW POST
    if (isset($_POST['post_new'], $_POST['channel'])) {
        if (!check_rate_limit($ip)) $error = "Rate limit exceeded.";
        elseif ((int)$_POST['captcha'] !== ($_SESSION['c1'] + $_SESSION['c2'])) $error = "Invalid CAPTCHA.";
        elseif (strlen($_POST['content']) > MAX_CONTENT_LENGTH) $error = "Content too long.";
        else {
            $c_name = $_POST['channel'];
            $p_file = POSTS_DIR . preg_replace('/[^a-z0-9_-]/i', '', $c_name) . '.json';
            $posts = file_exists($p_file) ? json_decode(file_get_contents($p_file), true) : [];
            
            $img_path = '';
            if (!empty($_FILES['image']['tmp_name']) && is_uploaded_file($_FILES['image']['tmp_name'])) {
                $img_path = image_to_ascii($_FILES['image']['tmp_name']);
            }

            $new_post = [
                'id' => uniqid('p', true),
                'content' => htmlspecialchars($_POST['content'], ENT_QUOTES),
                'link' => filter_var($_POST['link'] ?? '', FILTER_VALIDATE_URL) ?: '',
                'image' => $img_path,
                'timestamp' => date('c'), // ISO 8601
                'alpha_id' => get_poster_identity($ip), // PRIVACY AWARE ID
                'user_session' => $user_id,
                'comments' => []
            ];
            array_unshift($posts, $new_post);
            save_channel_posts($c_name, $posts);
            header("Location: ?channel=" . urlencode($c_name) . "#" . $new_post['id']); exit;
        }
    }

    // NEW COMMENT (Recursive insertion)
    if (isset($_POST['post_comment'], $_POST['channel'], $_POST['parent_id'])) {
        // ... (Same rate limit/captcha checks as above) ...
        if (!check_rate_limit($ip)) $error = "Rate limit exceeded.";
        elseif ((int)$_POST['captcha'] !== ($_SESSION['c1'] + $_SESSION['c2'])) $error = "Invalid CAPTCHA.";
        else {
             $c_name = $_POST['channel'];
             $p_file = POSTS_DIR . preg_replace('/[^a-z0-9_-]/i', '', $c_name) . '.json';
             if (file_exists($p_file)) {
                 $posts = json_decode(file_get_contents($p_file), true);
                 
                 $img_path = '';
                 if (!empty($_FILES['image']['tmp_name']) && is_uploaded_file($_FILES['image']['tmp_name'])) {
                     $img_path = image_to_ascii($_FILES['image']['tmp_name']);
                 }

                 $new_comment = [
                     'id' => uniqid('c', true),
                     'content' => htmlspecialchars($_POST['content'], ENT_QUOTES),
                     'link' => filter_var($_POST['link'] ?? '', FILTER_VALIDATE_URL) ?: '',
                     'image' => $img_path,
                     'timestamp' => date('c'),
                     'alpha_id' => get_poster_identity($ip),
                     'user_session' => $user_id,
                     'comments' => []
                 ];

                 // Recursive finder/inserter
                 $inserted = false;
                 $insert_comment = function(&$items, $target_id, $level) use (&$insert_comment, $new_comment, &$inserted) {
                     if ($level > 3) return; // Max depth 3
                     foreach ($items as &$item) {
                         if ($item['id'] === $target_id) {
                             $item['comments'][] = $new_comment;
                             $inserted = true; return;
                         }
                         $insert_comment($item['comments'], $target_id, $level + 1);
                         if ($inserted) return;
                     }
                 };
                 
                 $insert_comment($posts, $_POST['parent_id'], 0);
                 if ($inserted) {
                     save_channel_posts($c_name, $posts);
                     header("Location: ?channel=" . urlencode($c_name) . "#" . $new_comment['id']); exit;
                 } else {
                     $error = "Parent post/comment not found (might have been deleted).";
                 }
             }
        }
    }
}

// CAPTCHA REGEN
$_SESSION['c1'] = rand(2, 9); $_SESSION['c2'] = rand(2, 9);

// --- OPEN GRAPH & SEO ---
$og = [
    'title' => $config['site_name'],
    'desc' => $config['site_slogan'],
    'url' => (isset($_SERVER['HTTPS']) ? "https" : "http") . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]",
    'image' => '' // Add a default logo URL here if you have one
];
// If viewing a specific post, try to find it for better OG tags
if (isset($_GET['channel'], $_GET['post_id'])) {
    $p_file = POSTS_DIR . preg_replace('/[^a-z0-9_-]/i', '', $_GET['channel']) . '.json';
    if (file_exists($p_file)) {
        $posts = json_decode(file_get_contents($p_file), true);
        foreach ($posts as $p) {
            if ($p['id'] === $_GET['post_id']) {
                $og['title'] = $config['site_name'] . " - Post by " . $p['alpha_id'];
                $og['desc'] = substr($p['content'], 0, 200) . '...';
                if ($p['image']) $og['image'] = (isset($_SERVER['HTTPS']) ? "https" : "http") . "://$_SERVER[HTTP_HOST]/" . $p['image'];
                break;
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $og['title']; ?></title>
    <meta property="og:title" content="<?php echo htmlspecialchars($og['title'], ENT_QUOTES); ?>">
    <meta property="og:description" content="<?php echo htmlspecialchars($og['desc'], ENT_QUOTES); ?>">
    <meta property="og:url" content="<?php echo $og['url']; ?>">
    <meta property="og:image" content="<?php echo $og['image']; ?>">
    <meta property="og:type" content="website">
    <link rel="stylesheet" href="styles.css">
    <style>
        /* DYNAMIC THEME INJECTION */
        :root {
            --bg-color: <?php echo $config['theme']['bg_color']; ?>;
            --text-color: <?php echo $config['theme']['text_color']; ?>;
            --link-color: <?php echo $config['theme']['link_color']; ?>;
            --button-bg: <?php echo $config['theme']['button_bg']; ?>;
            --button-text: <?php echo $config['theme']['button_text']; ?>;
            --container-bg: <?php echo $config['theme']['container_bg']; ?>;
            --border-color: <?php echo $config['theme']['border_color']; ?>;
            --input-bg: <?php echo $config['theme']['input_bg']; ?>;
            --input-text: <?php echo $config['theme']['input_text']; ?>;
            --error-color: #ff5555; /* Fallback or add to theme options */
        }
        /* JS Highlighting */
        .highlight { animation: pulse 2s ease-out; border-color: var(--link-color) !important; }
        @keyframes pulse { 0% { background-color: var(--button-bg); } 100% { background-color: transparent; } }
        /* Admin Tabs */
        .admin-tabs { display: flex; margin-bottom: 1rem; border-bottom: 2px solid var(--border-color); }
        .admin-tab { padding: 10px 20px; cursor: pointer; background: var(--button-bg); color: var(--button-text); margin-right: 5px; border: 2px solid var(--border-color); border-bottom: none; opacity: 0.7; }
        .admin-tab.active { opacity: 1; font-weight: bold; background: var(--container-bg); }
        .tab-pane { display: none; padding: 20px; background: rgba(0,0,0,0.1); border: 2px solid var(--border-color); border-top: none; }
        .tab-pane.active { display: block; }
    </style>
    <script>
        document.addEventListener("DOMContentLoaded", () => {
            // Localize Timestamps
            document.querySelectorAll('[data-timestamp]').forEach(el => {
                try { el.textContent = new Date(el.getAttribute('data-timestamp')).toLocaleString(); } catch(e){}
            });
            // Highlight active post/comment from URL hash
            if(window.location.hash) {
                const target = document.getElementById(window.location.hash.substring(1));
                if(target) { target.scrollIntoView(); target.classList.add('highlight'); }
            }
            // Admin Tabs
            window.openTab = function(evt, tabName) {
                document.querySelectorAll('.tab-pane').forEach(p => p.style.display = "none");
                document.querySelectorAll('.admin-tab').forEach(t => t.classList.remove("active"));
                document.getElementById(tabName).style.display = "block";
                evt.currentTarget.classList.add("active");
            }
        });
        function copyLink(id) {
            const url = window.location.href.split('#')[0] + '#' + id;
            navigator.clipboard.writeText(url).then(() => alert("Link copied: " + url));
        }
    </script>
</head>
<body>
<div class="container">
    <div class="nav">
        <a href="index.php">Home</a>
        <a href="about.php">About</a>
        <a href="changelog.html">Changelog</a>
        <?php if($is_admin): ?><a href="?logout" style="float:right; border-color: var(--error-color);">Logout Admin</a><?php endif; ?>
    </div>

    <div class="header">
        <h1><a href="index.php"><?php echo $config['site_name']; ?></a></h1>
        <h3><?php echo $config['site_slogan']; ?></h3>
        <form method="post" action="index.php" class="form-group" style="max-width: 400px; margin: 1rem auto;">
            <input type="text" name="search" placeholder="Search all channels..." value="<?php echo htmlspecialchars($_POST['search']??'', ENT_QUOTES); ?>">
            <button type="submit">Search</button>
        </form>
    </div>

    <?php if (!empty($error)): ?>
        <div class="error" style="padding: 1rem; border: 2px solid var(--error-color); margin-bottom: 1rem;">
            <strong>Error:</strong> <?php echo $error; ?>
        </div>
    <?php endif; ?>
    <?php if (!empty($success)): ?>
        <div style="padding: 1rem; border: 2px solid green; color: green; margin-bottom: 1rem;">
            <?php echo $success; ?>
        </div>
    <?php endif; ?>

    <?php
    if (isset($_POST['search']) && !empty(trim($_POST['search']))) {
        // --- SEARCH RESULTS VIEW ---
        $query = trim($_POST['search']);
        echo "<h2>Search Results for: " . htmlspecialchars($query) . "</h2><ul>";
        $found = 0;
        foreach ($channels as $c) {
            $p_file = POSTS_DIR . preg_replace('/[^a-z0-9_-]/i', '', $c['name']) . '.json';
            if (!file_exists($p_file)) continue;
            $posts = json_decode(file_get_contents($p_file), true) ?: [];
            
            $search_walker = function($items, $c_name, $post_id) use (&$search_walker, $query, &$found) {
                foreach ($items as $item) {
                    if (stripos($item['content'], $query) !== false) {
                        echo "<li>Found in <strong>{$c_name}</strong>: <a href='?channel=".urlencode($c_name)."#{$item['id']}'>".substr(htmlspecialchars($item['content']), 0, 100)."...</a></li>";
                        $found++; if($found > 50) return;
                    }
                    if (!empty($item['comments'])) $search_walker($item['comments'], $c_name, $post_id ?: $item['id']);
                }
            };
            $search_walker($posts, $c['name'], null);
            if ($found > 50) break;
        }
        if ($found === 0) echo "<li>No results found.</li>";
        echo "</ul><p><a href='index.php'>Back to Home</a></p>";

    } elseif (isset($_GET['channel'])) {
        // --- SINGLE CHANNEL VIEW ---
        $c_name = $_GET['channel'];
        $channel_data = array_filter($channels, fn($c) => $c['name'] === $c_name);
        if (empty($channel_data)) {
            echo "<h2>Channel not found.</h2>";
        } else {
            $c_info = reset($channel_data);
            $views = update_channel_views($c_name, $_SERVER['REMOTE_ADDR']);
            
            echo "<h2>" . htmlspecialchars($c_info['name']) . "</h2>";
            echo "<p>" . htmlspecialchars($c_info['description']) . "</p>";
            if(!empty($c_info['disclosure'])) echo "<p><small><em>" . htmlspecialchars($c_info['disclosure']) . "</em></small></p>";
            echo "<p class='channel-stats'>Total Views: {$views['total']} | Unique Visitors: {$views['unique']} | <a href='rss.php?channel=".urlencode($c_name)."'>RSS Feed</a></p>";

            // Post Form
            ?>
            <div class="post-form" style="background: rgba(0,0,0,0.1); padding: 1rem; margin-bottom: 2rem; border: 2px solid var(--border-color);">
                <h3>Create New Post</h3>
                <form method="post" enctype="multipart/form-data">
                    <input type="hidden" name="channel" value="<?php echo htmlspecialchars($c_name); ?>">
                    <div class="form-group"><textarea name="content" rows="5" required placeholder="Share your thoughts..." maxlength="<?php echo MAX_CONTENT_LENGTH; ?>"></textarea></div>
                    <div class="form-group" style="display:flex; gap: 1rem; flex-wrap: wrap;">
                        <div><label>Image (will be ASCII'd):</label><input type="file" name="image" accept="image/*"></div>
                        <div style="flex-grow:1"><label>Optional Link:</label><input type="url" name="link" placeholder="https://..."></div>
                        <div><label>Captcha: <?php echo $_SESSION['c1'] . ' + ' . $_SESSION['c2']; ?> = </label><input type="number" name="captcha" style="width: 60px;" required></div>
                    </div>
                    <button type="submit" name="post_new">Submit Post</button>
                </form>
            </div>
            <?php

            // Display Posts
            $p_file = POSTS_DIR . preg_replace('/[^a-z0-9_-]/i', '', $c_name) . '.json';
            if (file_exists($p_file)) {
                $posts = json_decode(file_get_contents($p_file), true) ?: [];
                
                // Recursive Comment Renderer
                function render_comments($comments, $channel, $post_id, $level, $is_admin, $user_id) {
                    if (empty($comments)) return;
                    foreach ($comments as $c) {
                        $margin = $level * 20;
                        $bg = ($level % 2 === 0) ? 'rgba(0,0,0,0.05)' : 'rgba(0,0,0,0.02)';
                        echo "<div class='comment' id='{$c['id']}' style='margin-left:{$margin}px; background: $bg; padding: 10px; border-left: 2px solid var(--border-color); margin-top: 10px;'>";
                        echo "<div class='meta'><small><strong>{$c['alpha_id']}</strong> &bull; <span data-timestamp='{$c['timestamp']}'>{$c['timestamp']}</span> &bull; <a href='javascript:void(0)' onclick=\"copyLink('{$c['id']}')\">#</a></small>";
                         // Delete button for admin or owner
                         if ($is_admin || ($c['user_session']??'') === $user_id) {
			echo " <form method='post' onsubmit='return confirm(\"Delete this?\");' style='display:inline; float:right;'><input type='hidden' name='channel' value='".htmlspecialchars($channel)."'><input type='hidden' name='post_id' value='{$c['id']}'><button type='submit' name='delete_post' style='padding: 2px 5px; font-size: 0.7rem;'>DEL</button></form>";
			}                        
			echo "</div>";
                        if (!empty($c['image'])) echo "<img src='{$c['image']}' class='ascii-art' alt='ASCII'>";
                        echo "<p>" . nl2br($c['content']) . "</p>";
                        if (!empty($c['link'])) echo "<p><a href='{$c['link']}' target='_blank'>[External Link]</a></p>";
                        
                        // Reply Form (max depth 3)
                        if ($level < 3) {
                             echo "<details><summary>Reply</summary><form method='post' enctype='multipart/form-data' style='margin-top:10px;'>
                                <input type='hidden' name='channel' value='".htmlspecialchars($channel)."'>
                                <input type='hidden' name='parent_id' value='{$c['id']}'>
                                <textarea name='content' rows='3' required placeholder='Reply...'></textarea>
                                <div style='font-size: 0.8rem; margin-top:5px;'>
                                    Captcha: {$_SESSION['c1']}+{$_SESSION['c2']}=<input type='number' name='captcha' style='width:40px' required>
                                    Img: <input type='file' name='image' accept='image/*'>
                                </div>
                                <button type='submit' name='post_comment' style='margin-top:5px;'>Reply</button>
                            </form></details>";
                        }
                        render_comments($c['comments'] ?? [], $channel, $post_id, $level + 1, $is_admin, $user_id);
                        echo "</div>";
                    }
                }

                echo "<div class='posts'>";
                foreach ($posts as $post) {
                    echo "<div class='post' id='{$post['id']}'>";
                    echo "<div class='post-header'><strong>{$post['alpha_id']}</strong> &bull; <span data-timestamp='{$post['timestamp']}'>{$post['timestamp']}</span> &bull; <a href='javascript:void(0)' onclick=\"copyLink('{$post['id']}')\">#ID</a>";
                    if ($is_admin || ($post['user_session']??'') === $user_id) {
                        echo " <form method='post' onsubmit='return confirm(\"Delete this?\");' style='display:inline; float:right;'><input type='hidden' name='channel' value='".htmlspecialchars($c_name)."'><input type='hidden' name='post_id' value='{$post['id']}'><button type='submit' name='delete_post' style='padding: 2px 5px; font-size: 0.8rem;'>DELETE</button></form>";
                    }
                    echo "</div>";
                    if (!empty($post['image'])) echo "<img src='{$post['image']}' class='ascii-art' alt='ASCII'>";
                    echo "<div class='post-content'>" . nl2br($post['content']) . "</div>";
                    if (!empty($post['link'])) echo "<div class='post-link'><a href='{$post['link']}' target='_blank'>{$post['link']}</a></div>";
                    
                    echo "<div class='post-actions'><details><summary>Reply to Thread</summary>
                        <form method='post' enctype='multipart/form-data' style='margin-top:10px; padding:10px; border:1px dashed var(--border-color);'>
                            <input type='hidden' name='channel' value='".htmlspecialchars($c_name)."'>
                            <input type='hidden' name='parent_id' value='{$post['id']}'>
                            <textarea name='content' rows='3' required></textarea>
                            <div style='margin-top:5px;'>Captcha: {$_SESSION['c1']}+{$_SESSION['c2']}=<input type='number' name='captcha' style='width:50px' required> | Optional Img: <input type='file' name='image'></div>
                            <button type='submit' name='post_comment'>Post Reply</button>
                        </form>
                    </details></div>";
                    
                    render_comments($post['comments'] ?? [], $c_name, $post['id'], 1, $is_admin, $user_id);
                    echo "</div><hr>";
                }
                echo "</div>";
            }
        }

    } else {
        // --- HOME VIEW (Channel List) ---
        echo "<div class='site-description'><h2>Welcome</h2><p>" . nl2br($config['site_description'] ?? 'Welcome to our platform.') . "</p></div>";
        
        // Simple Stats
        $total_posts = 0;
        foreach ($channels as $c) {
            $p_file = POSTS_DIR . preg_replace('/[^a-z0-9_-]/i', '', $c['name']) . '.json';
            if (file_exists($p_file)) $total_posts += count(json_decode(file_get_contents($p_file), true) ?: []);
        }
        echo "<p class='site-stats'>Total Posts across all channels: $total_posts</p>";

        echo "<h2>Available Channels</h2><div class='channels'>";
        foreach ($channels as $c) {
            $safe_name = preg_replace('/[^a-z0-9_-]/i', '', $c['name']);
            $p_file = POSTS_DIR . $safe_name . '.json';
            $c_posts = file_exists($p_file) ? count(json_decode(file_get_contents($p_file),true)??[]) : 0;
            echo "<div class='channel-card'>";
            echo "<h3><a href='?channel=" . urlencode($c['name']) . "'>" . htmlspecialchars($c['name']) . "</a></h3>";
            echo "<p>" . htmlspecialchars($c['description']) . "</p>";
            echo "<p class='channel-stats'>Posts: $c_posts</p>";
            echo "</div>";
        }
        echo "</div>";
    }
    ?>

    <?php if ($is_admin): ?>
        <div id="admin-panel" style="margin-top: 50px; border-top: 5px solid var(--border-color); padding-top: 20px;">
            <h2>Admin Control Panel</h2>
            <div class="admin-tabs">
                <div class="admin-tab active" onclick="openTab(event, 'tab-gen')">General</div>
                <div class="admin-tab" onclick="openTab(event, 'tab-theme')">Theme</div>
                <div class="admin-tab" onclick="openTab(event, 'tab-about')">About Page</div>
                <div class="admin-tab" onclick="openTab(event, 'tab-chan')">Channels</div>
            </div>

            <div id="tab-gen" class="tab-pane active">
                <form method="post">
                    <h3>Main Settings</h3>
                    <div class="form-group"><label>Site Name:</label><input type="text" name="site_name" value="<?php echo $config['site_name']; ?>" required></div>
                    <div class="form-group"><label>Slogan:</label><input type="text" name="site_slogan" value="<?php echo $config['site_slogan']; ?>" required></div>
                    <div class="form-group"><label>Home Page Description:</label><textarea name="site_description" rows="5" required><?php echo $config['site_description'] ?? ''; ?></textarea></div>
                    <div class="form-group">
                        <label>Poster Privacy:</label>
                        <select name="privacy_mode">
                            <option value="hash" <?php echo ($config['privacy_mode'] === 'hash') ? 'selected' : ''; ?>>Hashed IDs (Unique Tripcodes)</option>
                            <option value="anonymous" <?php echo ($config['privacy_mode'] === 'anonymous') ? 'selected' : ''; ?>>Anonymous (No IDs displayed)</option>
                        </select>
                    </div>
                    <div class="form-group" style="border-top: 1px solid var(--border-color); padding-top:10px;">
                        <label>Change Admin Password (leave empty to keep current):</label>
                        <input type="password" name="new_password" placeholder="New Password">
                    </div>
                    <button type="submit" name="update_settings">Save Settings</button>
                </form>
            </div>

            <div id="tab-theme" class="tab-pane">
                <form method="post">
                    <h3>Customize Colors</h3>
                    <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 15px;">
                        <?php foreach ($config['theme'] as $key => $val): ?>
                            <label style="display:flex; justify-content:space-between;">
                                <?php echo ucwords(str_replace('_', ' ', $key)); ?>
                                <input type="color" name="theme_<?php echo $key; ?>" value="<?php echo $val; ?>">
                            </label>
                        <?php endforeach; ?>
                    </div>
                    <button type="submit" name="update_theme" style="margin-top: 20px;">Update Theme</button>
                </form>
            </div>

            <div id="tab-about" class="tab-pane">
                <?php $about = json_decode(file_get_contents(DATA_DIR . 'about.json'), true); ?>
                <form method="post">
                    <h3>Edit 'About' Page Content</h3>
                    <div class="form-group"><label>Page Title:</label><input type="text" name="about_title" value="<?php echo htmlspecialchars($about['title']); ?>" required></div>
                    <div class="form-group">
                        <label>HTML Content:</label>
                        <textarea name="about_content" rows="15" style="font-family: monospace;"><?php echo htmlspecialchars($about['content']); ?></textarea>
                    </div>
                    <button type="submit" name="update_about">Save About Page</button>
                </form>
            </div>

            <div id="tab-chan" class="tab-pane">
                <h3>Existing Channels</h3>
                <ul>
                    <?php foreach ($channels as $idx => $c): ?>
                        <li>
                            <strong><?php echo htmlspecialchars($c['name']); ?></strong> 
                            <form method="post" onsubmit="return confirm('Delete channel and ALL its posts? This cannot be undone.');" style="display:inline;">
                                <input type="hidden" name="channel_index" value="<?php echo $idx; ?>">
                                <button type="submit" name="delete_channel" style="background: var(--error-color); color: white; padding: 2px 5px; font-size: 0.8rem;">Delete</button>
                            </form>
                            <br><small><?php echo htmlspecialchars($c['description']); ?></small>
                        </li>
                    <?php endforeach; ?>
                </ul>
                <hr>
                <h3>Create New Channel</h3>
                <form method="post">
                    <div class="form-group"><input type="text" name="channel_name" placeholder="Channel Name" required></div>
                    <div class="form-group"><textarea name="channel_desc" placeholder="Description" required></textarea></div>
                    <div class="form-group"><textarea name="channel_disc" placeholder="Legal Disclosure / Rules (Optional)"></textarea></div>
                    <button type="submit" name="create_channel">Create Channel</button>
                </form>
            </div>
        </div>
    <?php endif; ?>

    <?php if (!$is_admin): ?>
        <div class="admin-login-footer" style="margin-top: 50px; opacity: 0.5; text-align: right;">
            <details>
                <summary>Admin</summary>
                <form method="post" style="margin-top: 10px; padding: 10px; background: rgba(0,0,0,0.2); display: inline-block; text-align: left;">
                    <input type="password" name="password" placeholder="Password" required>
                    <button type="submit" name="admin_login">Login</button>
                </form>
            </details>
        </div>
    <?php endif; ?>

    <div class="footer">
        <p><?php echo $config['site_name']; ?> &copy; <?php echo date('Y'); ?>. Powered by Queschan.</p>
    </div>
</div>
</body>
</html>
