<?php
declare(strict_types=1);

// Configuration File Path
const DATA_DIR = 'data/';
const CONFIG_FILE = DATA_DIR . 'config.json';

// Ensure data directory exists
if (!is_dir(DATA_DIR) && !mkdir(DATA_DIR, 0755, true)) {
    die("Error: Could not create data directory. Check permissions.");
}

// Check if already set up
if (file_exists(CONFIG_FILE)) {
    die("Site is already configured. Please delete 'data/config.json' if you need to run setup again, or delete this setup.php file for security.");
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $pass1 = $_POST['admin_pass'] ?? '';
    $pass2 = $_POST['admin_pass_confirm'] ?? '';

    if (empty($pass1) || strlen($pass1) < 8) {
        $error = "Password must be at least 8 characters.";
    } elseif ($pass1 !== $pass2) {
        $error = "Passwords do not match.";
    } else {
        // Default Theme
        $default_theme = [
            'bg_color' => '#3C2F2F',
            'text_color' => '#FFB300', // Adjusted to Amber for better contrast by default based on your CSS
            'link_color' => '#af2e03',
            'button_bg' => '#FFB300',
            'button_text' => '#af2e03',
            'container_bg' => '#FFB300',
            'border_color' => '#3C2F2F',
            'input_bg' => '#3C2F2F',
            'input_text' => '#FFF300',
        ];

        // Initial Configuration
        $config = [
            'site_name' => htmlspecialchars($_POST['site_name'] ?? 'Queschan', ENT_QUOTES, 'UTF-8'),
            'site_slogan' => htmlspecialchars($_POST['site_slogan'] ?? "'Q' is to Question", ENT_QUOTES, 'UTF-8'),
            'admin_hash' => password_hash($pass1, PASSWORD_DEFAULT),
            'privacy_mode' => $_POST['privacy_mode'] ?? 'hash', // 'hash' or 'anonymous'
            'hmac_secret' => bin2hex(random_bytes(32)), // Secret for IP hashing
            'theme' => $default_theme
        ];

        if (file_put_contents(CONFIG_FILE, json_encode($config, JSON_PRETTY_PRINT | JSON_THROW_ON_ERROR))) {
            $success = "Setup successful! Please delete setup.php and <a href='index.php'>go to your new site</a>.";
            
            // Initialize about page content if it doesn't exist
            $about_file = DATA_DIR . 'about.json';
            if (!file_exists($about_file)) {
                 $default_about = [
                    'title' => 'About ' . $config['site_name'],
                    'content' => '<p>This platform is dedicated to questioning narratives and fostering open discourse.</p><h2>Our Motto</h2><p>We Champion Freedom, Restore Autonomy.<br>Expect Clarity.</p><h2>Platform Guidelines</h2><ul><li>Use this space solely for questioning narratives and constructive dialogue.</li><li>Image uploads are converted to ASCII art.</li><li>Illegal content is strictly prohibited.</li></ul>'
                ];
                file_put_contents($about_file, json_encode($default_about, JSON_PRETTY_PRINT));
            }
        } else {
            $error = "Failed to write configuration file.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Queschan Setup</title>
    <style>
        body { font-family: 'Courier New', monospace; background: #333; color: #FFB300; padding: 20px; }
        .container { max-width: 600px; margin: 0 auto; background: #444; padding: 20px; border: 2px solid #FFB300; }
        input, select, button { display: block; width: 100%; margin-bottom: 15px; padding: 10px; box-sizing: border-box; font-family: inherit; }
        button { background: #FFB300; color: #333; border: none; cursor: pointer; font-weight: bold; }
        .error { color: #ff5555; }
        .success { color: #55ff55; }
    </style>
</head>
<body>
    <div class="container">
        <h1>Initial Setup</h1>
        <?php if ($error): ?><p class="error"><?php echo $error; ?></p><?php endif; ?>
        <?php if ($success): ?><p class="success"><?php echo $success; ?></p><?php endif; ?>

        <?php if (empty($success)): ?>
        <form method="post">
            <label>Site Name:</label>
            <input type="text" name="site_name" value="Queschan" required>
            
            <label>Site Slogan:</label>
            <input type="text" name="site_slogan" value="'Q' is to Question" required>

            <label>Privacy Mode:</label>
            <select name="privacy_mode">
                <option value="hash">Hash IPs (Default - Unique but anonymous IDs)</option>
                <option value="anonymous">Anonymous (Do not store/display any IDs)</option>
            </select>

            <label>Admin Password (min 8 chars):</label>
            <input type="password" name="admin_pass" required>

            <label>Confirm Admin Password:</label>
            <input type="password" name="admin_pass_confirm" required>

            <button type="submit">Install</button>
        </form>
        <?php endif; ?>
    </div>
</body>
</html>
