<?php
require_once 'config.php';

// Check if setup is completed
if (!isSetupCompleted()) {
    header('Location: setup.php');
    exit;
}

startAdminSession();

// Handle logout
if (isset($_GET['logout'])) {
    logoutAdmin();
    header('Location: admin.php');
    exit;
}

// Handle login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';

    if (verifyAdmin($username, $password)) {
        setAdminLoggedIn($username);
        header('Location: admin.php');
        exit;
    } else {
        $loginError = 'Invalid username or password';
    }
}

// Handle configuration update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_config']) && isAdminLoggedIn()) {
    $siteTitle = trim($_POST['site_title'] ?? '');
    $siteDescription = trim($_POST['site_description'] ?? '');
    $aboutContent = $_POST['about_content'] ?? '';
    $footerText = trim($_POST['footer_text'] ?? '');

    $errors = [];

    if (empty($siteTitle)) {
        $errors[] = 'Site title is required';
    }
    if (empty($siteDescription)) {
        $errors[] = 'Site description is required';
    }
    if (empty($footerText)) {
        $errors[] = 'Footer text is required';
    }

    if (empty($errors)) {
        try {
            updateSiteConfig($siteTitle, $siteDescription, $aboutContent, $footerText);
            $updateSuccess = true;
        } catch (Exception $e) {
            $errors[] = 'Update failed: ' . $e->getMessage();
        }
    }
}

// Get current configuration
$config = getSiteConfig();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - <?php echo htmlspecialchars($config['site_title']); ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Courier New', monospace;
            background-color: #000;
            color: #FFA500;
            padding: 20px;
            line-height: 1.6;
        }

        .container {
            max-width: 900px;
            margin: 0 auto;
            background-color: #111;
            padding: 30px;
            border-radius: 8px;
            border: 2px solid #FFA500;
        }

        h1 {
            margin-bottom: 20px;
            color: #FFA500;
            text-align: center;
        }

        h2 {
            margin: 30px 0 15px 0;
            color: #FFD700;
            border-bottom: 2px solid #333;
            padding-bottom: 10px;
        }

        .success {
            background-color: #1a4d1a;
            border: 2px solid #00ff00;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            color: #00ff00;
        }

        .error {
            background-color: #4d1a1a;
            border: 2px solid #ff0000;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            color: #ff0000;
        }

        .info {
            background-color: #1a1a4d;
            border: 2px solid #4444ff;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            color: #8888ff;
        }

        .form-group {
            margin-bottom: 20px;
        }

        label {
            display: block;
            margin-bottom: 5px;
            color: #FFA500;
            font-weight: bold;
        }

        input[type="text"],
        input[type="password"],
        textarea {
            width: 100%;
            padding: 10px;
            background-color: #000;
            color: #FFA500;
            border: 2px solid #FFA500;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }

        textarea {
            min-height: 200px;
            resize: vertical;
        }

        input:focus,
        textarea:focus {
            outline: none;
            border-color: #FFD700;
        }

        button {
            background-color: #FFA500;
            color: #000;
            padding: 12px 30px;
            border: none;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
        }

        button:hover {
            background-color: #FFD700;
        }

        .btn-secondary {
            background-color: #666;
            color: #fff;
        }

        .btn-secondary:hover {
            background-color: #888;
        }

        .note {
            font-size: 12px;
            color: #888;
            margin-top: 5px;
        }

        a {
            color: #FFD700;
            text-decoration: none;
        }

        a:hover {
            text-decoration: underline;
        }

        .header-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #333;
        }

        .nav-links {
            display: flex;
            gap: 20px;
        }

        .login-container {
            max-width: 400px;
            margin: 100px auto;
        }

        .preview {
            background-color: #000;
            border: 2px solid #333;
            padding: 15px;
            margin-top: 10px;
            border-radius: 4px;
            min-height: 50px;
        }

        .preview-label {
            color: #888;
            font-size: 12px;
            margin-bottom: 5px;
        }

        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 30px;
        }

        .button-group button {
            flex: 1;
        }
    </style>
</head>
<body>
    <?php if (!isAdminLoggedIn()): ?>
        <!-- Login Form -->
        <div class="container login-container">
            <h1>Admin Login</h1>

            <?php if (isset($loginError)): ?>
                <div class="error">
                    <?php echo htmlspecialchars($loginError); ?>
                </div>
            <?php endif; ?>

            <form method="POST" action="">
                <div class="form-group">
                    <label for="username">Username:</label>
                    <input type="text" id="username" name="username" required autofocus>
                </div>

                <div class="form-group">
                    <label for="password">Password:</label>
                    <input type="password" id="password" name="password" required>
                </div>

                <button type="submit" name="login">Login</button>
            </form>

            <div style="margin-top: 20px; text-align: center;">
                <a href="index.php">← Back to main site</a>
            </div>
        </div>
    <?php else: ?>
        <!-- Admin Dashboard -->
        <div class="container">
            <div class="header-bar">
                <div>
                    <h1>Admin Panel</h1>
                    <div style="color: #888; font-size: 14px;">
                        Logged in as: <strong><?php echo htmlspecialchars($_SESSION['admin_username']); ?></strong>
                    </div>
                </div>
                <div class="nav-links">
                    <a href="index.php">View Site</a>
                    <a href="?logout=1">Logout</a>
                </div>
            </div>

            <?php if (isset($updateSuccess) && $updateSuccess): ?>
                <div class="success">
                    ✓ Configuration updated successfully!
                </div>
            <?php endif; ?>

            <?php if (isset($errors) && !empty($errors)): ?>
                <div class="error">
                    <strong>Errors:</strong>
                    <ul style="margin-top: 10px; padding-left: 20px;">
                        <?php foreach ($errors as $error): ?>
                            <li><?php echo htmlspecialchars($error); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <div class="info">
                <strong>Site Customization:</strong> Use the form below to customize your site's appearance and content.
                All changes take effect immediately.
            </div>

            <form method="POST" action="" id="configForm">
                <h2>General Settings</h2>

                <div class="form-group">
                    <label for="site_title">Site Title:</label>
                    <input type="text" id="site_title" name="site_title"
                           value="<?php echo htmlspecialchars($config['site_title']); ?>"
                           required>
                    <div class="note">Appears in browser title, page header, and meta tags</div>
                </div>

                <div class="form-group">
                    <label for="site_description">Site Description:</label>
                    <input type="text" id="site_description" name="site_description"
                           value="<?php echo htmlspecialchars($config['site_description']); ?>"
                           required>
                    <div class="note">Used for meta tags and social media sharing (Open Graph)</div>
                </div>

                <h2>About Section</h2>

                <div class="form-group">
                    <label for="about_content">About Content:</label>
                    <textarea id="about_content" name="about_content"><?php echo htmlspecialchars($config['about_content']); ?></textarea>
                    <div class="note">
                        Supports line breaks, special characters, and HTML tags.<br>
                        This content appears below the main header on search pages.
                    </div>
                    <div class="preview-label">Preview:</div>
                    <div class="preview" id="aboutPreview"></div>
                </div>

                <h2>Footer</h2>

                <div class="form-group">
                    <label for="footer_text">Footer Text:</label>
                    <input type="text" id="footer_text" name="footer_text"
                           value="<?php echo htmlspecialchars($config['footer_text']); ?>"
                           required>
                    <div class="note">Copyright notice or footer information</div>
                </div>

                <div class="button-group">
                    <button type="submit" name="update_config">Save Changes</button>
                    <button type="button" class="btn-secondary" onclick="window.location.href='index.php'">
                        Cancel
                    </button>
                </div>
            </form>
        </div>

        <script>
            // Live preview for about content
            const aboutTextarea = document.getElementById('about_content');
            const aboutPreview = document.getElementById('aboutPreview');

            function updatePreview() {
                const content = aboutTextarea.value;
                // Convert line breaks to <br> tags but keep HTML
                const formatted = content.replace(/\n/g, '<br>');
                aboutPreview.innerHTML = formatted || '<em style="color: #666;">Preview will appear here...</em>';
            }

            aboutTextarea.addEventListener('input', updatePreview);

            // Initialize preview
            updatePreview();
        </script>
    <?php endif; ?>
</body>
</html>
