<?php
// Configuration file paths
define('CONFIG_DIR', __DIR__ . '/config_data');
define('SITE_CONFIG_FILE', CONFIG_DIR . '/site_config.json');
define('ADMIN_FILE', CONFIG_DIR . '/admin.json');

// Ensure config directory exists
if (!file_exists(CONFIG_DIR)) {
    mkdir(CONFIG_DIR, 0755, true);
}

// Get site configuration
function getSiteConfig() {
    if (!file_exists(SITE_CONFIG_FILE)) {
        // Return default config
        return [
            'site_title' => 'Query Gram',
            'site_description' => 'Query records and easily share results found.',
            'about_content' => 'This is a full-text search interface for document records.',
            'footer_text' => 'Copyright ' . date('Y')
        ];
    }

    $json = file_get_contents(SITE_CONFIG_FILE);
    $config = json_decode($json, true);

    if (!$config) {
        // Return default if JSON is invalid
        return [
            'site_title' => 'Query Gram',
            'site_description' => 'Query records and easily share results found.',
            'about_content' => 'This is a full-text search interface for document records.',
            'footer_text' => 'Copyright ' . date('Y')
        ];
    }

    return $config;
}

// Update site configuration
function updateSiteConfig($title, $description, $about, $footer) {
    $config = [
        'site_title' => $title,
        'site_description' => $description,
        'about_content' => $about,
        'footer_text' => $footer,
        'updated_at' => date('Y-m-d H:i:s')
    ];

    $json = json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    return file_put_contents(SITE_CONFIG_FILE, $json) !== false;
}

// Check if setup is completed
function isSetupCompleted() {
    return file_exists(ADMIN_FILE) && file_exists(SITE_CONFIG_FILE);
}

// Get admin credentials
function getAdminCredentials() {
    if (!file_exists(ADMIN_FILE)) {
        return null;
    }

    $json = file_get_contents(ADMIN_FILE);
    return json_decode($json, true);
}

// Create admin user
function createAdmin($username, $password) {
    $admin = [
        'username' => $username,
        'password_hash' => password_hash($password, PASSWORD_DEFAULT),
        'created_at' => date('Y-m-d H:i:s')
    ];

    $json = json_encode($admin, JSON_PRETTY_PRINT);
    return file_put_contents(ADMIN_FILE, $json) !== false;
}

// Verify admin credentials
function verifyAdmin($username, $password) {
    $admin = getAdminCredentials();

    if (!$admin) {
        return false;
    }

    if ($admin['username'] !== $username) {
        return false;
    }

    return password_verify($password, $admin['password_hash']);
}

// Session management
function startAdminSession() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
}

function isAdminLoggedIn() {
    startAdminSession();
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

function setAdminLoggedIn($username) {
    startAdminSession();
    $_SESSION['admin_logged_in'] = true;
    $_SESSION['admin_username'] = $username;
}

function logoutAdmin() {
    startAdminSession();
    session_unset();
    session_destroy();
}
