<?php
require_once 'config.php';

// Get site configuration
$config = getSiteConfig();

// Handle file download requests
if (isset($_GET['download']) && !empty($_GET['download'])) {
    $downloadFile = basename($_GET['download']); // Sanitize filename
    $textDir = './text/';
    $filePath = $textDir . $downloadFile;

    // Check if file exists and is within text directory
    if (file_exists($filePath) && is_file($filePath)) {
        $ext = pathinfo($downloadFile, PATHINFO_EXTENSION);
        if ($ext === 'txt' || $ext === 'srt') {
            // Set headers for download
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . $downloadFile . '"');
            header('Content-Length: ' . filesize($filePath));
            header('Cache-Control: no-cache');

            // Output file
            readfile($filePath);
            exit;
        }
    }

    // If we get here, file not found or invalid
    header('HTTP/1.0 404 Not Found');
    echo "File not found or invalid.";
    exit;
}

header('Content-Type: text/html; charset=UTF-8');

// Parse URL parameters
$query = isset($_GET['q']) ? urldecode($_GET['q']) : '';
$before = max(0, (int)($_GET['b'] ?? 2));
$after = max(0, (int)($_GET['a'] ?? 2));
$selectedFiles = $_GET['files'] ?? [];
$page = max(1, (int)($_GET['page'] ?? 1));
$perPage = 100; // Results per lazy-load batch (increased for better performance)
$caseSensitive = isset($_GET['cs']) && $_GET['cs'] === '1';

// AJAX request for lazy loading
$isAjax = !empty($_GET['ajax']) && $_GET['ajax'] === '1';

// Get list of text files
$textDir = './text/';
$files = array_diff(scandir($textDir), array('..', '.'));

// Parse SRT file into searchable entries
function parseSrtFile($filePath) {
    $content = file_get_contents($filePath);
    $entries = [];

    // Split by double newlines to get each subtitle block
    $blocks = preg_split('/\n\s*\n/', trim($content));

    foreach ($blocks as $block) {
        $lines = explode("\n", trim($block));
        if (count($lines) >= 3) {
            $number = trim($lines[0]);
            $timeframe = trim($lines[1]);
            // Concatenate all remaining lines as text (some entries span multiple lines)
            $text = implode(' ', array_slice($lines, 2));

            $entries[] = [
                'number' => $number,
                'timeframe' => $timeframe,
                'text' => $text,
                'combined' => "$number | $timeframe | $text" // For searching
            ];
        }
    }

    return $entries;
}

// Improved search logic with proper boolean evaluation
function evaluateQuery($text, $query, $caseSensitive = false) {
    // Tokenize the query
    $tokens = preg_split('/\s+/', $query);
    $stack = [];
    $operators = [];

    $precedence = [
        'NOT' => 3,
        'AND' => 2,
        'NAND' => 2,
        'OR' => 1,
        'XOR' => 1,
        'NOR' => 1
    ];

    $i = 0;
    while ($i < count($tokens)) {
        $token = $tokens[$i];
        $upperToken = strtoupper($token);

        // Handle NOT as prefix operator
        if ($upperToken === 'NOT' && $i + 1 < count($tokens)) {
            $i++;
            $nextToken = $tokens[$i];
            if (!in_array(strtoupper($nextToken), ['AND', 'OR', 'XOR', 'NOR', 'NAND', 'NOT'])) {
                // Apply NOT to the next term
                if ($caseSensitive) {
                    $contains = strpos($text, $nextToken) === false; // Inverted, case-sensitive
                } else {
                    $contains = stripos($text, $nextToken) === false; // Inverted, case-insensitive
                }
                $stack[] = $contains;
            }
        }
        // Handle binary operators
        elseif (in_array($upperToken, ['AND', 'OR', 'XOR', 'NOR', 'NAND'])) {
            // Apply operators based on precedence
            while (!empty($operators) &&
                   isset($precedence[$operators[count($operators) - 1]]) &&
                   $precedence[$operators[count($operators) - 1]] >= $precedence[$upperToken]) {
                applyOperator($stack, array_pop($operators));
            }
            $operators[] = $upperToken;
        }
        // Handle search terms
        else {
            if ($caseSensitive) {
                $contains = strpos($text, $token) !== false; // Case-sensitive
            } else {
                $contains = stripos($text, $token) !== false; // Case-insensitive
            }
            $stack[] = $contains;
        }

        $i++;
    }

    // Apply remaining operators
    while (!empty($operators)) {
        applyOperator($stack, array_pop($operators));
    }

    return !empty($stack) ? $stack[0] : false;
}

function applyOperator(&$stack, $operator) {
    if (count($stack) < 2) return;

    $right = array_pop($stack);
    $left = array_pop($stack);

    switch ($operator) {
        case 'AND':
            $stack[] = $left && $right;
            break;
        case 'OR':
            $stack[] = $left || $right;
            break;
        case 'XOR':
            $stack[] = $left xor $right;
            break;
        case 'NOR':
            $stack[] = !($left || $right);
            break;
        case 'NAND':
            $stack[] = !($left && $right);
            break;
    }
}

// Only output HTML if not AJAX
if (!$isAjax) {
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title><?php echo htmlspecialchars($config['site_title']); ?></title>
    <meta property="og:title" content="<?php echo htmlspecialchars($config['site_title']); ?>">
    <meta property="og:description" content="<?php echo htmlspecialchars($config['site_description']); ?>">
    <meta property="og:image" content="<?php echo htmlspecialchars('http://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['REQUEST_URI']) . '/logo.png'); ?>">
    <meta property="og:url" content="<?php echo htmlspecialchars('http://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']); ?>">
    <meta property="og:type" content="website">
    <link rel="icon" type="image/x-icon" href="./fav.ico">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body {
            font-family: 'Courier New', Courier, monospace;
            margin: 0;
            padding: 0;
            background: #000;
            color: #FFA500;
            font-size: 16px;
        }
        header {
            text-align: center;
            padding: 1.5rem;
            background: #000;
            color: #FFA500;
            border-bottom: 2px solid #FFA500;
        }
        h1 {
            margin: 0;
            font-size: 2rem;
        }
        .container {
            margin: 0 auto;
            padding: 1rem;
            max-width: 100%;
            box-sizing: border-box;
        }
        .main-table {
            display: flex;
            flex-direction: column;
            gap: 1rem;
            margin: 1rem 0;
        }
        .main-table > div {
            padding: 1rem;
            border: 2px solid #FFA500;
            color: #FFA500;
            background: #000;
            border-radius: 8px;
        }
        .query-area {
            flex: 1 1 auto;
        }
        .legend-area {
            flex: 1 1 auto;
        }
        table.inner-table {
            width: 100%;
            border-collapse: collapse;
            border: 2px solid #FFA500;
            font-size: 1rem;
            background: #000;
            border-radius: 8px;
        }
        th, td.inner-table {
            padding: 1rem;
            text-align: left;
            border: 1px solid #FFA500;
            color: #FFA500;
        }
        th {
            background: #222;
        }
        tr:nth-child(even) {
            background: #111;
        }
        tr:nth-child(odd) {
            background: #000;
        }
        footer {
            text-align: center;
            padding: 1.5rem;
            background: #000;
            color: #FFA500;
            border-top: 2px solid #FFA500;
        }
        a {
            background-color: #FFA500;
            color: #000;
            padding: 12px 24px;
            text-decoration: none;
            display: inline-block;
            border: none;
            border-radius: 4px;
            transition: background-color 0.3s, color 0.3s;
            font-size: 1.1rem;
        }
        a:hover {
            background-color: #000;
            color: #FFA500;
            border: 1px solid #FFA500;
        }
        a.match-link {
            padding: 4px 8px;
            font-size: 0.9rem;
            margin-right: 10px;
        }
        form {
            background: #FFA500;
            padding: 1rem;
            border-radius: 8px;
        }
        input, textarea, select {
            font-family: 'Courier New', Courier, monospace;
            color: #000;
            background: orange;
            border: 1px solid #000;
            font-size: 1.2rem;
            padding: 12px;
            margin: 6px 0;
            border-radius: 4px;
            width: 100%;
            box-sizing: border-box;
        }
        input[type="text"] {
            height: 70px;
        }
        input[type="number"] {
            width: 90px;
            height: 70px;
            margin-right: 10px;
        }
        input[type="submit"] {
            height: 70px;
            cursor: pointer;
            font-weight: bold;
        }
        form label {
            color: #000;
            font-size: 1.1rem;
            display: flex;
            align-items: center;
            gap: 8px;
            margin: 6px 0;
        }
        form label input[type="checkbox"] {
            width: auto;
            margin: 0;
        }
        .context-settings {
            display: flex;
            align-items: center;
            gap: 15px;
            margin: 6px 0;
            flex-wrap: wrap;
        }
        .file-list {
            margin: 0.5rem 0;
            max-height: 200px;
            overflow-y: auto;
        }
        .download-link {
            padding: 2px 8px !important;
            font-size: 0.9rem !important;
            background-color: #FFA500 !important;
            color: #000 !important;
            text-decoration: none;
            border-radius: 4px;
            display: inline-block;
            transition: background-color 0.3s;
        }
        .download-link:hover {
            background-color: #FFD700 !important;
            color: #000 !important;
        }
        li { list-style: none; margin: 0; }
        p { margin: 0; }
        .line-number {
            display: inline-block;
            width: 60px;
            text-align: right;
            margin-right: 15px;
            color: #FFA500;
        }
        .highlight {
            background-color: #FFD700;
            color: #000;
            padding: 2px 4px;
        }
        h3 {
            margin: 0 0 1rem 0;
            font-size: 1.5rem;
        }
        .match {
            margin: 1rem 0;
            padding: 1rem;
            border: 2px solid #FFA500;
            border-radius: 8px;
            background: #111;
        }
        .about-section {
            padding: 1rem;
            margin: 0 0 1rem 0;
            border: 2px solid #FFA500;
            border-radius: 8px;
            line-height: 1.8;
            width: 100%;
            max-width: 100%;
            box-sizing: border-box;
        }
        .loading {
            text-align: center;
            padding: 2rem;
            color: #FFD700;
            font-size: 1.2rem;
        }
        .load-more {
            text-align: center;
            margin: 2rem 0;
        }
        #results {
            min-height: 200px;
            width: 100%;
            max-width: 100%;
            box-sizing: border-box;
        }
        .match {
            width: 100%;
            max-width: 100%;
            box-sizing: border-box;
            word-wrap: break-word;
            overflow-wrap: break-word;
        }
        .admin-link {
            float: right;
            font-size: 0.8rem;
            padding: 8px 16px;
        }
        @media (min-width: 768px) {
            .main-table {
                flex-direction: row;
                justify-content: space-between;
            }
            .main-table > div {
                flex: 1 1 45%;
                max-width: 48%;
            }
            .container {
                margin: 0 auto;
                padding: 1.5rem;
                max-width: 100%;
            }
            h1 {
                font-size: 2.5rem;
            }
        }
        @media (min-width: 1400px) {
            .container {
                padding: 2rem;
            }
        }
        .action-buttons {
            margin: 1rem 0;
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
        }
        .action-btn {
            padding: 10px 20px !important;
            font-size: 1rem !important;
        }
        @media print {
            header, form, .admin-link, .load-more, .action-buttons,
            .main-table, .about-section, .match-link, .download-link, .shareable-url {
                display: none !important;
            }
            body {
                background: white;
                color: black;
            }
            footer {
                background: white;
                color: black;
                border-top: 2px solid black;
            }
            .match {
                border: 2px solid black;
                background: white;
                color: black;
                page-break-inside: avoid;
                margin: 1rem 0;
            }
            .highlight {
                background-color: #ffff99;
                color: black;
            }
            .line-number {
                color: #666;
            }
            #results h3 {
                color: black;
            }
            #results p {
                color: black;
            }
        }
    </style>
</head>
<body>
    <header>
        <a href="admin.php" class="admin-link">Admin</a>
        <h1><?php echo htmlspecialchars($config['site_title']); ?></h1>
    </header>
    <div class="container">
        <div class="about-section">
            <?php echo nl2br($config['about_content']); ?>
        </div>
        <div class="main-table">
            <div class="query-area">
                <form method="GET" action="" id="searchForm">
                    <h3>Text Query</h3>
                    <input type="text" name="q"
                           placeholder="e.g., apple AND orange OR banana NOT pear"
                           value="<?php echo htmlspecialchars($query); ?>">

                    <h3>Context Settings</h3>
                    <div class="context-settings">
                        <label>Above: <input type="number" name="b" value="<?php echo $before; ?>" min="0"></label>
                        <label>Below: <input type="number" name="a" value="<?php echo $after; ?>" min="0"></label>
                    </div>

                    <div style="margin: 12px 0;">
                        <label><input type="checkbox" name="cs" value="1" <?php echo $caseSensitive ? 'checked' : ''; ?>> Case Sensitive</label>
                    </div>

                    <h3>Select Files</h3>
                    <div class="file-list">
                        <label><input type="checkbox" onclick="toggleAll(this)" checked> Check/Uncheck All</label>
                        <?php
                        foreach ($files as $file) {
                            $ext = pathinfo($file, PATHINFO_EXTENSION);
                            if ($ext === 'txt' || $ext === 'srt') {
                                $checked = empty($selectedFiles) || in_array($file, $selectedFiles) ? 'checked' : '';
                                $downloadUrl = '?download=' . urlencode($file);
                                echo "<label style='display: flex; align-items: center; gap: 8px;'>";
                                echo "<input type='checkbox' name='files[]' value='$file' $checked>";
                                echo "<span style='flex: 1;'>" . htmlspecialchars($file) . "</span>";
                                echo "<a href='$downloadUrl' class='download-link' title='Download this file' style='padding: 2px 6px; font-size: 0.8rem;'>⬇</a>";
                                echo "</label>";
                            }
                        }
                        ?>
                    </div>
                    <input type="submit" value="Search">
                </form>
            </div>
            <div class="legend-area">
                <h3>Operand Legend</h3>
                <table class="inner-table">
                    <tr><th>Gate</th><th>Description</th></tr>
                    <tr><td class="inner-table">AND</td><td class="inner-table">Returns true if both inputs are true</td></tr>
                    <tr><td class="inner-table">OR</td><td class="inner-table">Returns true if at least one input is true</td></tr>
                    <tr><td class="inner-table">XOR</td><td class="inner-table">Returns true if exactly one input is true</td></tr>
                    <tr><td class="inner-table">NOR</td><td class="inner-table">Returns true if both inputs are false</td></tr>
                    <tr><td class="inner-table">NOT</td><td class="inner-table">Inverts the next term (prefix operator)</td></tr>
                    <tr><td class="inner-table">NAND</td><td class="inner-table">Returns true if at least one input is false</td></tr>
                </table>
            </div>
        </div>

        <!-- Results -->
        <div id="results">
<?php
}

// Process search with streaming
if ($query) {
    // Debug logging for AJAX requests
    if ($isAjax) {
        error_log("AJAX Request - Page: $page, Query: $query, Files: " . json_encode($selectedFiles));
    }

    if (!$isAjax) {
        echo "<h3>Results for: " . htmlspecialchars($query) . ($caseSensitive ? " (Case Sensitive)" : "") . "</h3>";
        $baseUrl = "?q=" . urlencode($query) . "&b=$before&a=$after";
        if ($caseSensitive) {
            $baseUrl .= "&cs=1";
        }
        foreach ($selectedFiles as $file) {
            $baseUrl .= "&files[]=" . urlencode($file);
        }
        echo "<p class='shareable-url'>Shareable URL: <a href='$baseUrl'>" . htmlspecialchars($baseUrl) . "</a></p>";
        echo "<div class='action-buttons'>";
        echo "<a href='#' onclick='printResults(); return false;' class='action-btn'>Print Results</a>";
        echo "<a href='#' onclick='downloadResults(); return false;' class='action-btn'>Download Results</a>";
        echo "</div>";

        // Flush output immediately
        if (ob_get_level() > 0) {
            ob_flush();
        }
        flush();
    }

    $matchCount = 0;
    $displayCount = 0;
    $skip = ($page - 1) * $perPage;
    $hasMore = false;
    $stopProcessing = false;

    foreach ($files as $file) {
        if ($stopProcessing) break;
        if (!empty($selectedFiles) && !in_array($file, $selectedFiles)) continue;

        $fileExt = pathinfo($file, PATHINFO_EXTENSION);
        if ($fileExt !== 'txt' && $fileExt !== 'srt') continue;

        // Handle SRT files differently
        if ($fileExt === 'srt') {
            $srtEntries = parseSrtFile($textDir . $file);

            foreach ($srtEntries as $i => $entry) {
                if (evaluateQuery($entry['combined'], $query, $caseSensitive)) {
                    $matchCount++;

                    // Skip results for pagination
                    if ($matchCount <= $skip) {
                        continue;
                    }

                    // Check if we've displayed enough for this page
                    if ($displayCount >= $perPage) {
                        $hasMore = true;
                        $stopProcessing = true;
                        break;
                    }

                    $displayCount++;
                    $anchor = "match$matchCount";
                    echo "<div class='match' id='$anchor'>";
                    echo "<a href='#$anchor' class='match-link'>#$matchCount</a> ";
                    echo "<strong>File:</strong> " . htmlspecialchars($file) . "<br>";
                    echo "<strong>Entry:</strong> " . htmlspecialchars($entry['number']) . " | ";
                    echo "<strong>Time:</strong> " . htmlspecialchars($entry['timeframe']) . "<br>";

                    // Show context before (if available)
                    $contextStart = max(0, $i - $before);
                    for ($j = $contextStart; $j < $i; $j++) {
                        echo "<span class='line-number'>" . htmlspecialchars($srtEntries[$j]['number']) . ":</span>";
                        echo htmlspecialchars($srtEntries[$j]['timeframe']) . " - ";
                        echo htmlspecialchars($srtEntries[$j]['text']) . "<br>";
                    }

                    // Show matched entry (highlighted)
                    echo "<span class='line-number'>" . htmlspecialchars($entry['number']) . ":</span>";
                    echo "<span class='highlight'>" . htmlspecialchars($entry['timeframe']) . " - ";
                    echo htmlspecialchars($entry['text']) . "</span><br>";

                    // Show context after (if available)
                    $contextEnd = min(count($srtEntries) - 1, $i + $after);
                    for ($j = $i + 1; $j <= $contextEnd; $j++) {
                        echo "<span class='line-number'>" . htmlspecialchars($srtEntries[$j]['number']) . ":</span>";
                        echo htmlspecialchars($srtEntries[$j]['timeframe']) . " - ";
                        echo htmlspecialchars($srtEntries[$j]['text']) . "<br>";
                    }
                    echo "</div>";

                    // Flush output after each result for streaming
                    if (ob_get_level() > 0) {
                        ob_flush();
                    }
                    flush();
                }
            }
        } else {
            // Handle TXT files (original logic)
            $lines = file($textDir . $file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            if (!$lines) continue;

            foreach ($lines as $i => $line) {
                if (evaluateQuery($line, $query, $caseSensitive)) {
                    $matchCount++;

                    // Skip results for pagination
                    if ($matchCount <= $skip) {
                        continue;
                    }

                    // Check if we've displayed enough for this page
                    if ($displayCount >= $perPage) {
                        // We found at least one more result, so there are more pages
                        $hasMore = true;
                        $stopProcessing = true;
                        break;
                    }

                    $displayCount++;
                    $anchor = "match$matchCount";
                    echo "<div class='match' id='$anchor'>";
                    echo "<a href='#$anchor' class='match-link'>#$matchCount</a> ";
                    echo "<strong>File:</strong> " . htmlspecialchars($file) . "<br>";

                    for ($j = max(0, $i - $before); $j < $i; $j++) {
                        $lineNum = $j + 1;
                        echo "<span class='line-number'>$lineNum:</span>" . htmlspecialchars($lines[$j]) . "<br>";
                    }

                    $matchLineNum = $i + 1;
                    echo "<span class='line-number'>$matchLineNum:</span><span class='highlight'>" . htmlspecialchars($line) . "</span><br>";

                    for ($j = $i + 1; $j <= min(count($lines) - 1, $i + $after); $j++) {
                        $lineNum = $j + 1;
                        echo "<span class='line-number'>$lineNum:</span>" . htmlspecialchars($lines[$j]) . "<br>";
                    }
                    echo "</div>";

                    // Flush output after each result for streaming
                    if (ob_get_level() > 0) {
                        ob_flush();
                    }
                    flush();
                }
            }
        }
    }

    if ($displayCount === 0 && !$isAjax) {
        echo "<p>No matches found.</p>";
    } elseif ($displayCount === 0 && $isAjax) {
        echo "<p><!-- No results on this page. Total matches: $matchCount, Skip: $skip, PerPage: $perPage --></p>";
    }

    // Output pagination info for AJAX
    if ($isAjax) {
        echo "<!-- AJAX Response: Page $page, Query: '$query', Skip: $skip, TotalMatches: $matchCount, Displayed: $displayCount, HasMore: " . ($hasMore ? 'true' : 'false') . " -->";
        echo "<script>window.hasMoreResults = " . ($hasMore ? 'true' : 'false') . "; window.currentPage = $page;</script>";
    }
}

if (!$isAjax) {
?>
        </div>

        <?php if ($query && $hasMore): ?>
        <div class="load-more">
            <a href="#" id="loadMoreBtn" onclick="loadMore(); return false;">Load More Results</a>
            <div id="loading" class="loading" style="display: none;">Loading...</div>
        </div>
        <?php endif; ?>
    </div>

    <footer>
        <?php echo htmlspecialchars($config['footer_text']); ?>
    </footer>

    <script>
    let currentPage = <?php echo $page; ?>;
    let isLoading = false;
    window.hasMoreResults = <?php echo ($query && $hasMore) ? 'true' : 'false'; ?>;

    function toggleAll(checkbox) {
        const checkboxes = document.querySelectorAll('input[name="files[]"]');
        checkboxes.forEach(cb => cb.checked = checkbox.checked);
    }

    function loadMore() {
        if (isLoading) return;

        isLoading = true;
        const loadMoreBtn = document.getElementById('loadMoreBtn');
        const loadingDiv = document.getElementById('loading');

        if (loadMoreBtn) loadMoreBtn.style.display = 'none';
        if (loadingDiv) loadingDiv.style.display = 'block';

        const urlParams = new URLSearchParams(window.location.search);
        currentPage++;
        urlParams.set('page', currentPage);
        urlParams.set('ajax', '1');

        const fullUrl = '?' + urlParams.toString();
        console.log('AJAX Request URL:', fullUrl);
        console.log('Query param:', urlParams.get('q'));
        console.log('Files param:', urlParams.getAll('files[]'));

        fetch(fullUrl)
            .then(response => response.text())
            .then(html => {
                console.log('AJAX Response received, length:', html.length);
                console.log('First 500 chars:', html.substring(0, 500));

                const resultsDiv = document.getElementById('results');

                // Extract script tags to check for hasMoreResults
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = html;
                const scripts = tempDiv.getElementsByTagName('script');

                // Execute scripts to get hasMoreResults variable
                for (let script of scripts) {
                    eval(script.textContent);
                }

                console.log('After AJAX - hasMoreResults:', window.hasMoreResults, 'currentPage:', window.currentPage);

                // Remove scripts from HTML before appending
                while (tempDiv.querySelector('script')) {
                    tempDiv.querySelector('script').remove();
                }

                resultsDiv.insertAdjacentHTML('beforeend', tempDiv.innerHTML);

                isLoading = false;
                if (loadingDiv) loadingDiv.style.display = 'none';

                console.log('Content loaded, hasMoreResults:', window.hasMoreResults);

                if (window.hasMoreResults) {
                    if (loadMoreBtn) loadMoreBtn.style.display = 'inline-block';
                    // Trigger scroll check after content is loaded
                    setTimeout(checkScroll, 100);
                } else {
                    const loadMoreContainer = document.querySelector('.load-more');
                    if (loadMoreContainer) {
                        loadMoreContainer.innerHTML = '<p>All results loaded.</p>';
                    }
                }
            })
            .catch(error => {
                console.error('Error loading more results:', error);
                isLoading = false;
                if (loadingDiv) loadingDiv.style.display = 'none';
                if (loadMoreBtn) loadMoreBtn.style.display = 'inline-block';
                alert('Error loading more results. Please try again.');
            });
    }

    // Check if we need to load more based on scroll position
    function checkScroll() {
        console.log('checkScroll called - isLoading:', isLoading, 'hasMoreResults:', window.hasMoreResults);

        if (isLoading || !window.hasMoreResults) return;

        const loadMoreContainer = document.querySelector('.load-more');
        if (!loadMoreContainer) {
            console.log('No load-more container found');
            return;
        }

        const rect = loadMoreContainer.getBoundingClientRect();
        const windowHeight = window.innerHeight;

        console.log('Load more button top:', rect.top, 'Window height + 800:', windowHeight + 800);

        // Load more if the button is within 800px of viewport
        if (rect.top < windowHeight + 800) {
            console.log('Auto-loading more results...');
            loadMore();
        }
    }

    // Debounced scroll event
    let scrollTimeout;
    window.addEventListener('scroll', function() {
        if (scrollTimeout) clearTimeout(scrollTimeout);
        scrollTimeout = setTimeout(checkScroll, 150);
    });

    // Check on page load
    window.addEventListener('load', function() {
        setTimeout(checkScroll, 500);
    });

    document.querySelector('form').addEventListener('submit', function(e) {
        const textQuery = document.querySelector('input[name="q"]').value;
        const before = document.querySelector('input[name="b"]').value;
        const after = document.querySelector('input[name="a"]').value;
        const caseSensitive = document.querySelector('input[name="cs"]').checked;
        const files = Array.from(document.querySelectorAll('input[name="files[]"]:checked'))
                          .map(cb => 'files[]=' + encodeURIComponent(cb.value));

        if (textQuery) {
            window.location = '?q=' + encodeURIComponent(textQuery) +
                           '&b=' + before +
                           '&a=' + after +
                           (caseSensitive ? '&cs=1' : '') +
                           (files.length ? '&' + files.join('&') : '');
            e.preventDefault();
        }
    });

    function printResults() {
        window.print();
    }

    function downloadResults() {
        const resultsDiv = document.getElementById('results');
        if (!resultsDiv) {
            alert('No results to download');
            return;
        }

        // Get the query from URL params
        const urlParams = new URLSearchParams(window.location.search);
        const query = urlParams.get('q') || 'search';
        const caseSensitive = urlParams.get('cs') === '1';

        // Create formatted text content
        let textContent = '';
        textContent += '='.repeat(80) + '\n';
        textContent += 'SEARCH RESULTS\n';
        textContent += '='.repeat(80) + '\n\n';
        textContent += 'Query: ' + query + (caseSensitive ? ' (Case Sensitive)' : '') + '\n';
        textContent += 'Date: ' + new Date().toLocaleString() + '\n';
        textContent += '='.repeat(80) + '\n\n';

        // Extract all match divs
        const matches = resultsDiv.querySelectorAll('.match');
        matches.forEach((match, index) => {
            // Extract text content from each match
            const lines = match.innerText.split('\n');
            textContent += '-'.repeat(80) + '\n';
            textContent += 'Match #' + (index + 1) + '\n';
            textContent += '-'.repeat(80) + '\n';

            lines.forEach(line => {
                if (line.trim()) {
                    textContent += line + '\n';
                }
            });
            textContent += '\n';
        });

        textContent += '='.repeat(80) + '\n';
        textContent += 'End of Results (Total: ' + matches.length + ' matches)\n';
        textContent += '='.repeat(80) + '\n';

        // Create filename with date-time prefix and search terms suffix
        const now = new Date();
        const datePrefix = now.getFullYear() + '-' +
                          String(now.getMonth() + 1).padStart(2, '0') + '-' +
                          String(now.getDate()).padStart(2, '0') + '-' +
                          String(now.getHours()).padStart(2, '0') +
                          String(now.getMinutes()).padStart(2, '0') +
                          String(now.getSeconds()).padStart(2, '0');

        // Sanitize search terms for filename
        const searchTerms = query.replace(/[^a-zA-Z0-9]/g, '-').substring(0, 50);
        const filename = datePrefix + '-' + searchTerms + '.txt';

        // Create blob and download
        const blob = new Blob([textContent], { type: 'text/plain;charset=utf-8' });
        const link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(link.href);
    }
    </script>
</body>
</html>
<?php
}
?>
