<?php
require_once 'config.php';

// Check if setup is already completed
$setupCompleted = isSetupCompleted();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$setupCompleted) {
    $username = trim($_POST['username'] ?? 'admin');
    $password = trim($_POST['password'] ?? '');
    $siteTitle = trim($_POST['site_title'] ?? 'Query Gram');
    $siteDescription = trim($_POST['site_description'] ?? 'Query records and easily share results found.');
    $aboutContent = $_POST['about_content'] ?? 'This is a full-text search interface for document records.';
    $footerText = trim($_POST['footer_text'] ?? 'Copyright ' . date('Y'));

    $errors = [];

    // Validate inputs
    if (empty($username)) {
        $errors[] = 'Username is required';
    }
    if (empty($password)) {
        $errors[] = 'Password is required';
    }
    if (strlen($password) < 6) {
        $errors[] = 'Password must be at least 6 characters';
    }
    if (empty($siteTitle)) {
        $errors[] = 'Site title is required';
    }

    if (empty($errors)) {
        try {
            // Create admin user
            createAdmin($username, $password);

            // Save site configuration
            updateSiteConfig($siteTitle, $siteDescription, $aboutContent, $footerText);

            $success = true;
        } catch (Exception $e) {
            $errors[] = 'Setup failed: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Site Setup</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Courier New', monospace;
            background-color: #000;
            color: #FFA500;
            padding: 20px;
            line-height: 1.6;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #111;
            padding: 30px;
            border-radius: 8px;
            border: 2px solid #FFA500;
        }

        h1 {
            margin-bottom: 20px;
            color: #FFA500;
            text-align: center;
        }

        .success {
            background-color: #1a4d1a;
            border: 2px solid #00ff00;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            color: #00ff00;
        }

        .error {
            background-color: #4d1a1a;
            border: 2px solid #ff0000;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            color: #ff0000;
        }

        .form-group {
            margin-bottom: 20px;
        }

        label {
            display: block;
            margin-bottom: 5px;
            color: #FFA500;
            font-weight: bold;
        }

        input[type="text"],
        input[type="password"],
        textarea {
            width: 100%;
            padding: 10px;
            background-color: #000;
            color: #FFA500;
            border: 2px solid #FFA500;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }

        textarea {
            min-height: 150px;
            resize: vertical;
        }

        input:focus,
        textarea:focus {
            outline: none;
            border-color: #FFD700;
        }

        button {
            background-color: #FFA500;
            color: #000;
            padding: 12px 30px;
            border: none;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            width: 100%;
        }

        button:hover {
            background-color: #FFD700;
        }

        .info {
            background-color: #1a1a4d;
            border: 2px solid #4444ff;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            color: #8888ff;
        }

        .note {
            font-size: 12px;
            color: #888;
            margin-top: 5px;
        }

        a {
            color: #FFD700;
            text-decoration: none;
        }

        a:hover {
            text-decoration: underline;
        }

        .completed {
            text-align: center;
        }

        .completed h2 {
            color: #00ff00;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <?php if (isset($success) && $success): ?>
            <div class="success">
                <h2>✓ Setup Completed Successfully!</h2>
                <p>Your site has been configured. You can now:</p>
                <ul style="margin: 15px 0; padding-left: 20px;">
                    <li><a href="index.php">Visit the main search page</a></li>
                    <li><a href="admin.php">Login to the admin panel</a></li>
                </ul>
                <p><strong>Username:</strong> <?php echo htmlspecialchars($username); ?></p>
                <p class="note" style="color: #ff8800;">Please save your credentials securely!</p>
            </div>
        <?php elseif ($setupCompleted): ?>
            <div class="completed">
                <h1>Setup Already Completed</h1>
                <p style="margin: 20px 0;">This site has already been set up.</p>
                <p><a href="index.php">Go to main page</a> | <a href="admin.php">Go to admin panel</a></p>
            </div>
        <?php else: ?>
            <h1>Site Setup</h1>

            <?php if (isset($errors) && !empty($errors)): ?>
                <div class="error">
                    <strong>Errors:</strong>
                    <ul style="margin-top: 10px; padding-left: 20px;">
                        <?php foreach ($errors as $error): ?>
                            <li><?php echo htmlspecialchars($error); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <div class="info">
                <strong>Welcome!</strong> This is the initial setup for your search site.
                Please configure your admin credentials and site information below.
            </div>

            <form method="POST" action="">
                <div class="form-group">
                    <label for="username">Admin Username:</label>
                    <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($_POST['username'] ?? 'admin'); ?>" required>
                    <div class="note">Default: admin</div>
                </div>

                <div class="form-group">
                    <label for="password">Admin Password:</label>
                    <input type="password" id="password" name="password" required>
                    <div class="note">Minimum 6 characters</div>
                </div>

                <hr style="border: 1px solid #333; margin: 30px 0;">

                <div class="form-group">
                    <label for="site_title">Site Title:</label>
                    <input type="text" id="site_title" name="site_title" value="<?php echo htmlspecialchars($_POST['site_title'] ?? 'Query Gram'); ?>" required>
                    <div class="note">This appears in the browser title bar and page header</div>
                </div>

                <div class="form-group">
                    <label for="site_description">Site Description:</label>
                    <input type="text" id="site_description" name="site_description" value="<?php echo htmlspecialchars($_POST['site_description'] ?? 'Query records and easily share results found.'); ?>" required>
                    <div class="note">Used for meta tags and social media sharing</div>
                </div>

                <div class="form-group">
                    <label for="about_content">About Section:</label>
                    <textarea id="about_content" name="about_content"><?php echo htmlspecialchars($_POST['about_content'] ?? 'This is a full-text search interface for document records.

You can use logic gates in your queries:
- AND: Both terms must be present
- OR: Either term must be present
- NOT: Negate the next term
- XOR: Exactly one term must be present
- NAND: At least one term must be absent
- NOR: Both terms must be absent'); ?></textarea>
                    <div class="note">Supports line breaks and special characters. You can use HTML tags.</div>
                </div>

                <div class="form-group">
                    <label for="footer_text">Footer Text:</label>
                    <input type="text" id="footer_text" name="footer_text" value="<?php echo htmlspecialchars($_POST['footer_text'] ?? 'Copyright ' . date('Y')); ?>" required>
                    <div class="note">Copyright notice or footer information</div>
                </div>

                <button type="submit">Complete Setup</button>
            </form>
        <?php endif; ?>
    </div>
</body>
</html>
