#!/bin/bash
#
# ELF (Extremely Low Frequency) Chart Generator
# Generates pure black-on-white spectrogram for frequencies 100Hz and below
# from MTS, MP4, and WAV files using SoX, ffmpeg, ImageMagick, and GNUPlot.
# Output: Black = high magnitude, White = low magnitude
# Orientation: Time axis = horizontal, Frequency axis = vertical
#

set -e

# ── Defaults ──────────────────────────────────────────────────────────────────
OUTPUT_HEIGHT=1080
PIXELS_PER_SECOND=5
MAX_FREQ=100
SAMPLE_RATE=1000       # 1kHz gives Nyquist of 500Hz; 0–100Hz is 20% of display
MAX_OUTPUT_WIDTH=8000  # Avoids ImageMagick policy limits
BG_COLOR="white"
FG_COLOR="black"
INPUT_DIR="."
OUTPUT_DIR="."
INTERACTIVE=false

# ── Usage ─────────────────────────────────────────────────────────────────────
usage() {
    cat <<EOF
Usage: $(basename "$0") [OPTIONS] [FILE ...]

ELF (Extremely Low Frequency) Chart Generator
Produces a black-on-white FFT spectrogram from MTS, MP4, or WAV files,
clipped to 0–MAX_FREQ Hz with horizontal frequency markers and a video
filmstrip aligned to the time axis.

OPTIONS:
  -H, --height NUM         Chart height in pixels              (default: $OUTPUT_HEIGHT)
  -p, --pps NUM            Pixels per second on the time axis  (default: $PIXELS_PER_SECOND)
  -f, --max-freq HZ        Maximum frequency displayed         (default: $MAX_FREQ)
  -r, --sample-rate HZ     SoX resample rate (Nyquist = /2)   (default: $SAMPLE_RATE)
  -w, --max-width PX       Maximum output image width          (default: $MAX_OUTPUT_WIDTH)
  -d, --dir PATH           Directory to scan for media files   (default: current dir)
  -o, --output-dir PATH    Directory for generated PNG files   (default: current dir)
  -i, --interactive        Prompt for each setting (defaults shown in [])
  -h, --help               Show this help and exit

EXAMPLES:
  $(basename "$0")                          # Scan current dir, all defaults
  $(basename "$0") -i                       # Interactive setup
  $(basename "$0") -p 10 -H 720 clip.mp4   # Custom PPS and height for one file
  $(basename "$0") -d /mnt/footage -o /tmp/charts

SUPPORTED INPUT FORMATS: MTS, MP4, WAV
EOF
}

# ── Interactive prompt helper ─────────────────────────────────────────────────
# Usage: prompt_var VAR_NAME "Prompt text"
# Leaves the variable unchanged if the user presses Enter.
prompt_var() {
    local var_name="$1"
    local prompt_text="$2"
    local current_val="${!var_name}"
    local input
    read -r -p "  $prompt_text [$current_val]: " input
    if [ -n "$input" ]; then
        printf -v "$var_name" '%s' "$input"
    fi
}

# ── Parse arguments ───────────────────────────────────────────────────────────
POSITIONAL_ARGS=()

while [[ $# -gt 0 ]]; do
    case "$1" in
        -H|--height)        OUTPUT_HEIGHT="$2";      shift 2 ;;
        -p|--pps)           PIXELS_PER_SECOND="$2";  shift 2 ;;
        -f|--max-freq)      MAX_FREQ="$2";            shift 2 ;;
        -r|--sample-rate)   SAMPLE_RATE="$2";         shift 2 ;;
        -w|--max-width)     MAX_OUTPUT_WIDTH="$2";    shift 2 ;;
        -d|--dir)           INPUT_DIR="$2";           shift 2 ;;
        -o|--output-dir)    OUTPUT_DIR="$2";          shift 2 ;;
        -i|--interactive)   INTERACTIVE=true;         shift ;;
        -h|--help)          usage; exit 0 ;;
        -*)  echo "Unknown option: $1"; echo; usage; exit 1 ;;
        *)   POSITIONAL_ARGS+=("$1"); shift ;;
    esac
done

# ── Interactive mode ──────────────────────────────────────────────────────────
if $INTERACTIVE; then
    echo "ELF Chart Generator — Interactive Setup"
    echo "Press Enter to accept the default shown in [brackets]."
    echo ""
    prompt_var OUTPUT_HEIGHT     "Chart height (px)"
    prompt_var PIXELS_PER_SECOND "Pixels per second (time axis)"
    prompt_var MAX_FREQ          "Maximum frequency to chart (Hz)"
    prompt_var SAMPLE_RATE       "SoX resample rate (Hz; Nyquist = rate/2)"
    prompt_var MAX_OUTPUT_WIDTH  "Maximum output image width (px)"
    prompt_var INPUT_DIR         "Input directory"
    prompt_var OUTPUT_DIR        "Output directory"
    echo ""
fi

# ── Validate numeric settings ─────────────────────────────────────────────────
for var in OUTPUT_HEIGHT PIXELS_PER_SECOND MAX_FREQ SAMPLE_RATE MAX_OUTPUT_WIDTH; do
    val="${!var}"
    if ! [[ "$val" =~ ^[0-9]+$ ]] || [ "$val" -eq 0 ]; then
        echo "Error: $var must be a positive integer (got: '$val')"
        exit 1
    fi
done

# ── Check dependencies ────────────────────────────────────────────────────────
for cmd in sox ffmpeg gnuplot convert mogrify; do
    if ! command -v "$cmd" &> /dev/null; then
        echo "Error: $cmd is required but not installed."
        exit 1
    fi
done

# ── Temporary directory ───────────────────────────────────────────────────────
TMPDIR=$(mktemp -d)
trap "rm -rf $TMPDIR" EXIT

# ── Resolve output directory ──────────────────────────────────────────────────
mkdir -p "$OUTPUT_DIR"
OUTPUT_DIR=$(realpath "$OUTPUT_DIR")

# ── Collect input files ───────────────────────────────────────────────────────
if [ ${#POSITIONAL_ARGS[@]} -gt 0 ]; then
    FILES=("${POSITIONAL_ARGS[@]}")
else
    shopt -s nullglob nocaseglob
    FILES=("$INPUT_DIR"/*.mts "$INPUT_DIR"/*.mp4 "$INPUT_DIR"/*.wav)
    shopt -u nullglob nocaseglob
fi

if [ ${#FILES[@]} -eq 0 ]; then
    echo "No MTS, MP4, or WAV files found."
    exit 1
fi

echo "Found ${#FILES[@]} media file(s) to process."
echo "  Height: ${OUTPUT_HEIGHT}px  |  PPS: ${PIXELS_PER_SECOND}  |  Max freq: ${MAX_FREQ}Hz  |  Sample rate: ${SAMPLE_RATE}Hz"
echo "  Output dir: $OUTPUT_DIR"
echo ""

# ── Process each file ─────────────────────────────────────────────────────────
for FILE in "${FILES[@]}"; do
    echo "Processing: $FILE"

    BASENAME=$(basename "$FILE")
    NAME="${BASENAME%.*}"
    EXT="${BASENAME##*.}"
    EXT_LOWER=$(echo "$EXT" | tr '[:upper:]' '[:lower:]')

    WAV_FILE="$TMPDIR/${NAME}_temp.wav"
    RESAMPLED_FILE="$TMPDIR/${NAME}_resampled.wav"
    SPEC_RAW="$TMPDIR/${NAME}_sox_raw.png"
    SPEC_PNG="$TMPDIR/${NAME}_sox_spec.png"
    OUTPUT_FILE="$OUTPUT_DIR/${NAME}_elf_chart.png"

    # ── Step 1: Extract / convert audio to WAV ────────────────────────────────
    echo "  Extracting audio..."
    if [ "$EXT_LOWER" = "wav" ]; then
        cp "$FILE" "$WAV_FILE"
    else
        ffmpeg -y -i "$FILE" -vn -acodec pcm_s16le -ar 44100 -ac 1 "$WAV_FILE" 2>/dev/null
    fi

    # ── Step 2: Resample and low-pass filter for ELF ──────────────────────────
    echo "  Filtering for ELF frequencies (0–${MAX_FREQ}Hz)..."
    sox "$WAV_FILE" -r "$SAMPLE_RATE" "$RESAMPLED_FILE" lowpass "$MAX_FREQ"

    # ── Step 3: Get duration ──────────────────────────────────────────────────
    DURATION=$(sox "$RESAMPLED_FILE" -n stat 2>&1 | grep "Length" | awk '{print $3}')
    DURATION_INT=${DURATION%.*}
    if [ -z "$DURATION_INT" ] || [ "$DURATION_INT" -eq 0 ]; then
        DURATION_INT=1
    fi

    # ── Step 4: Calculate output dimensions ───────────────────────────────────
    EFFECTIVE_PPS=$PIXELS_PER_SECOND
    OUTPUT_WIDTH=$((DURATION_INT * EFFECTIVE_PPS))

    if [ "$OUTPUT_WIDTH" -gt "$MAX_OUTPUT_WIDTH" ]; then
        EFFECTIVE_PPS=$(echo "scale=4; $MAX_OUTPUT_WIDTH / $DURATION_INT" | bc)
        OUTPUT_WIDTH=$MAX_OUTPUT_WIDTH
        echo "  Long video detected — reducing to ${EFFECTIVE_PPS} px/sec"
    fi

    MIN_WIDTH=$((OUTPUT_HEIGHT + 100))   # Enforce landscape orientation
    if [ "$OUTPUT_WIDTH" -lt "$MIN_WIDTH" ]; then
        OUTPUT_WIDTH=$MIN_WIDTH
    fi

    echo "  Duration: ${DURATION_INT}s  |  Output: ${OUTPUT_WIDTH}x${OUTPUT_HEIGHT}px"

    # ── Step 5: Generate SoX spectrogram ─────────────────────────────────────
    echo "  Generating spectrogram with SoX..."
    sox "$RESAMPLED_FILE" -n spectrogram \
        -x "$OUTPUT_WIDTH" \
        -Y "$OUTPUT_HEIGHT" \
        -z 80 \
        -m -l -r \
        -o "$SPEC_RAW" 2>/dev/null || true

    if [ ! -f "$SPEC_RAW" ] || [ ! -s "$SPEC_RAW" ]; then
        echo "  Error: SoX failed to generate spectrogram — skipping $FILE."
        continue
    fi

    # Force exact pixel dimensions (remove any SoX padding)
    mogrify -limit memory 2GiB -limit map 4GiB -limit disk 8GiB \
        -resize "${OUTPUT_WIDTH}x${OUTPUT_HEIGHT}!" "$SPEC_RAW"

    cp "$SPEC_RAW" "$SPEC_PNG"

    # ── Step 6: Calculate frequency marker Y positions ────────────────────────
    # SoX spectrograms display 0 to Nyquist.  A small empirical offset corrects
    # for the way SoX aligns frequency bins within the image.
    NYQUIST=$((SAMPLE_RATE / 2))
    FREQ_OFFSET=10
    LINE_50HZ_Y=$(echo "scale=2; ((50 + $FREQ_OFFSET) / $NYQUIST) * $OUTPUT_HEIGHT" | bc)
    LINE_20HZ_Y=$(echo "scale=2; ((20 + $FREQ_OFFSET) / $NYQUIST) * $OUTPUT_HEIGHT" | bc)
    # Invert: image Y=0 is the top, frequency Y=0 is the bottom
    LINE_50HZ_Y_INV=$(echo "scale=2; $OUTPUT_HEIGHT - $LINE_50HZ_Y" | bc)
    LINE_20HZ_Y_INV=$(echo "scale=2; $OUTPUT_HEIGHT - $LINE_20HZ_Y" | bc)

    # ── Step 7: Annotate with ImageMagick (GNUPlot as fallback) ──────────────
    echo "  Adding frequency markers and labels..."
    if command -v convert &> /dev/null; then
        convert -limit memory 2GiB -limit map 4GiB -limit disk 8GiB \
            "$SPEC_PNG" \
            -fill "gray40" -stroke "gray40" -strokewidth 1 \
            -draw "line 0,${LINE_50HZ_Y_INV} ${OUTPUT_WIDTH},${LINE_50HZ_Y_INV}" \
            -fill "gray60" -stroke "gray60" -strokewidth 1 \
            -draw "line 0,${LINE_20HZ_Y_INV} ${OUTPUT_WIDTH},${LINE_20HZ_Y_INV}" \
            -fill "gray40" -pointsize 20 -gravity NorthWest \
            -annotate "+10+${LINE_50HZ_Y_INV}" "50Hz" \
            -fill "gray60" -pointsize 20 -gravity NorthWest \
            -annotate "+10+${LINE_20HZ_Y_INV}" "20Hz" \
            -fill "$FG_COLOR" -pointsize 24 -gravity NorthWest \
            -annotate "+10+30" "$NAME — HTTPS://ESUI.CC | ELF Chart (0–${MAX_FREQ}Hz)" \
            -fill "$FG_COLOR" -pointsize 16 -gravity SouthEast \
            -annotate "+10+10" "Time: ${EFFECTIVE_PPS}px/sec (horizontal)" \
            "$OUTPUT_FILE"
    else
        echo "  ImageMagick not found — falling back to GNUPlot for annotation."
        gnuplot <<GNUPLOT_SCRIPT
set terminal pngcairo size $OUTPUT_WIDTH,$OUTPUT_HEIGHT enhanced font 'Arial,14' background '$BG_COLOR'
set output '$OUTPUT_FILE'
unset border
unset tics
unset key
set lmargin at screen 0.0
set rmargin at screen 1.0
set tmargin at screen 1.0
set bmargin at screen 0.0
set xrange [0:$OUTPUT_WIDTH]
set yrange [0:$OUTPUT_HEIGHT]
set label 1 "$NAME — ELF Chart (0-${MAX_FREQ}Hz)" at 10,($OUTPUT_HEIGHT-30) left font ',18' tc rgb "$FG_COLOR"
set label 2 "Time: ${EFFECTIVE_PPS}px/sec (horizontal)" at ($OUTPUT_WIDTH-10),20 right font ',12' tc rgb "$FG_COLOR"
set label 3 "50Hz" at 50,$LINE_50HZ_Y_INV left font ',16' tc rgb "gray40"
set label 4 "20Hz" at 50,$LINE_20HZ_Y_INV left font ',16' tc rgb "gray60"
set arrow 1 from 0,$LINE_50HZ_Y_INV to $OUTPUT_WIDTH,$LINE_50HZ_Y_INV nohead lc rgb "gray40" lw 2 dt 2
set arrow 2 from 0,$LINE_20HZ_Y_INV to $OUTPUT_WIDTH,$LINE_20HZ_Y_INV nohead lc rgb "gray60" lw 2 dt 2
plot '$SPEC_PNG' binary filetype=png with rgbimage notitle
GNUPLOT_SCRIPT
    fi

    if [ ! -f "$OUTPUT_FILE" ]; then
        echo "  Warning: annotation step failed — using raw SoX spectrogram."
        cp "$SPEC_PNG" "$OUTPUT_FILE"
    fi

    # ── Step 8: Video filmstrip (skipped for WAV) ─────────────────────────────
    if [ "$EXT_LOWER" != "wav" ]; then
        echo "  Extracting video stills for filmstrip..."

        THUMB_WIDTH=72
        THUMB_INTERVAL_PX=72
        THUMB_INTERVAL_SEC=$(echo "scale=2; $THUMB_INTERVAL_PX / $EFFECTIVE_PPS" | bc)
        THUMB_HEIGHT=$((THUMB_WIDTH * 9 / 16))

        FILMSTRIP="$TMPDIR/${NAME}_filmstrip.png"
        THUMB_DIR="$TMPDIR/${NAME}_thumbs"
        mkdir -p "$THUMB_DIR"

        ffmpeg -y -i "$FILE" \
            -vf "fps=1/${THUMB_INTERVAL_SEC},scale=${THUMB_WIDTH}:${THUMB_HEIGHT}" \
            "$THUMB_DIR/thumb_%04d.png" 2>/dev/null

        THUMB_FILES=("$THUMB_DIR"/thumb_*.png)
        if [ ${#THUMB_FILES[@]} -gt 0 ]; then
            NUM_THUMBS=$((OUTPUT_WIDTH / THUMB_WIDTH))
            FILMSTRIP_THUMBS=()
            for ((i=0; i<NUM_THUMBS; i++)); do
                if [ $i -lt ${#THUMB_FILES[@]} ]; then
                    FILMSTRIP_THUMBS+=("${THUMB_FILES[$i]}")
                else
                    FILMSTRIP_THUMBS+=("${THUMB_FILES[-1]}")
                fi
            done

            convert -limit memory 2GiB -limit map 4GiB -limit disk 8GiB \
                "${FILMSTRIP_THUMBS[@]}" +append "$FILMSTRIP"

            mogrify -limit memory 2GiB -limit map 4GiB -limit disk 8GiB \
                -resize "${OUTPUT_WIDTH}x${THUMB_HEIGHT}!" "$FILMSTRIP"

            SPACER="$TMPDIR/${NAME}_spacer.png"
            convert -size "${OUTPUT_WIDTH}x2" xc:white "$SPACER"

            convert -limit memory 2GiB -limit map 4GiB -limit disk 8GiB \
                "$OUTPUT_FILE" "$SPACER" "$FILMSTRIP" -append "$OUTPUT_FILE"

            echo "  Filmstrip: ${#FILMSTRIP_THUMBS[@]} stills at ${THUMB_INTERVAL_SEC}s intervals"
        fi
    fi

    echo "  Done: $OUTPUT_FILE"
    echo ""
done

echo "Finished. Generated ELF charts for ${#FILES[@]} file(s)."
